/**
 * Color represented in HSV space
 */
export type ColorHSVT = {
    /**
     * hue component (0..360)
     */
    hue: number;
    /**
     * saturation component (0..100)
     */
    saturation: number;
    /**
     * value component (0..100)
     */
    value: number | null;
};
/**
 * Color represented in HSL space
 */
export type ColorHSLT = {
    /**
     * hue component (0..360)
     */
    hue: number;
    /**
     * saturation component (0..100)
     */
    saturation: number;
    /**
     * lightness component (0..100)
     */
    lightness: number;
};
/**
 * Color represented in CIE space
 */
export type ColorXYT = {
    /**
     * X component (0..1)
     */
    x: number;
    /**
     * Y component (0..1)
     */
    y: number;
};
/**
 * Color represented with red, green and blue components
 */
export type ColorRGBT = {
    /**
     * red component (0..1)
     */
    red: number;
    /**
     * green component (0..1)
     */
    green: number;
    /**
     * blue component (0..1)
     */
    blue: number;
};
export class ColorRGB {
    /**
     * Create RGB color from object
     * @param {ColorRGBT} rgb object with properties red, green and blue
     * @return {ColorRGB} new ColoRGB object
     */
    static fromObject(rgb: ColorRGBT): ColorRGB;
    /**
     * Create RGB color from hex string
     * @param {string} hex hex encoded RGB color
     * @return {ColorRGB} new ColoRGB object
     */
    static fromHex(hex: string): ColorRGB;
    /**
     * Create RGB color
     * @param {number} red
     * @param {number} green
     * @param {number} blue
     */
    constructor(red: number, green: number, blue: number);
    /** red component (0..1) */
    red: number;
    /** green component (0..1) */
    green: number;
    /** blue component (0..1) */
    blue: number;
    /**
     * Return this color with values rounded to given precision
     * @param {number} precision decimal places to round to
     * @return {ColorRGB}
     */
    rounded(precision: number): ColorRGB;
    /**
     * Convert to Object
     * @return {ColorRGBT} object with properties red, green and blue
     */
    toObject(): ColorRGBT;
    /**
     * Convert to HSV
     * @return {ColorHSV} color in HSV space
     */
    toHSV(): ColorHSV;
    /**
     * Convert to CIE
     * @return {ColorXY} color in CIE space
     */
    toXY(): ColorXY;
    /**
     * Returns color after sRGB gamma correction
     * @return {ColorRGB} corrected RGB
     */
    gammaCorrected(): ColorRGB;
    /**
     * Returns color after reverse sRGB gamma correction
     * @return {ColorRGB} raw RGB
     */
    gammaUncorrected(): ColorRGB;
    /**
     * Create hex string from RGB color
     * @return {ColorRGB} hex hex encoded RGB color
     */
    toHEX(): ColorRGB;
}
/**
 *  Class representing color in CIE space
 */
export class ColorXY {
    /**
     * Create CIE color from object
     * @param {ColorXYT} xy object with properties x and y
     * @return {ColorXY} new ColorXY object
     */
    static fromObject(xy: ColorXYT): ColorXY;
    /**
     * Create XY object from color temp in mireds
     * @param {number} mireds color temp in mireds
     * @return {ColorXY} color in XY space
     */
    static fromMireds(mireds: number): ColorXY;
    /**
     * Create CIE color
     * @param {number} x
     * @param {number} y
     */
    constructor(x: number, y: number);
    /** x component (0..1) */
    x: number;
    /** y component (0..1) */
    y: number;
    /**
     * Converts color in XY space to temperature in mireds
     * @return {number} color temp in mireds
     */
    toMireds(): number;
    /**
     * Converts CIE color space to RGB color space
     * From: https://github.com/usolved/cie-rgb-converter/blob/master/cie_rgb_converter.js
     * @return {ColorRGB}
     */
    toRGB(): ColorRGB;
    /**
     * Convert to HSV
     * @return {ColorHSV} color in HSV space
     */
    toHSV(): ColorHSV;
    /**
     * Return this color with value rounded to given precision
     * @param {number} precision decimal places to round to
     * @return {ColorXY}
     */
    rounded(precision: number): ColorXY;
    /**
     * Convert to object
     * @return {ColorXYT} object with properties x and y
     */
    toObject(): ColorXYT;
}
/**
 * Class representing color in HSV space
 */
export class ColorHSV {
    /**
     * Create HSV color from object
     * @param {object} hsv
     * @param {number} [hsv.hue]
     * @param {number} [hsv.saturation]
     * @param {number} [hsv.value]
     * @return {ColorHSV}
     */
    static fromObject(hsv: {
        hue?: number;
        saturation?: number;
        value?: number;
    }): ColorHSV;
    /**
     * Create HSV color from HSL
     * @param {ColorHSL} hsl color in HSL space
     * @return {ColorHSV} color in HSV space
     */
    static fromHSL(hsl: ColorHSL): ColorHSV;
    /**
     * Interpolates hue value based on correction map through ranged linear interpolation
     * @param {Nnmber} hue hue to be corrected
     * @param {Array} correctionMap array of hueIn -> hueOut mappings; example: [ {"in": 20, "out": 25}, {"in": 109, "out": 104}]
     * @return {number} corrected hue value
     */
    static interpolateHue(hue: Nnmber, correctionMap: any[]): number;
    /**
     * Applies hue interpolation if entity has hue correction data
     * @param {number} hue hue component of HSV color
     * @param {Object} meta entity meta object
     * @param {Object} meta.options meta object's options property
     * @param {Object} [meta.options.hue_correction] hue correction data
     * @return {number} corrected hue component of HSV color
     */
    static correctHue(hue: number, meta: {
        options: {
            hue_correction?: Object;
        };
    }): number;
    /**
     * Create color in HSV space
     * @param {?number} hue
     * @param {?number} [saturation=null]
     * @param {?number} [value=null]
     */
    constructor(hue: number | null, saturation?: number | null, value?: number | null);
    /** hue component (0..360) */
    hue: number;
    /** saturation component (0..100) */
    saturation: number;
    /** value component (0..100) */
    value: number;
    /**
     * Return this color with value rounded to given precision
     * @param {number} precision decimal places to round to
     * @return {ColorHSV}
     */
    rounded(precision: number): ColorHSV;
    /**
     * Convert to object
     * @param {boolean} short return h, s, v instead of hue, saturation, value
     * @param {boolean} includeValue omit v(alue) from return
     * @return {ColorHSVT}
     */
    toObject(short?: boolean, includeValue?: boolean): ColorHSVT;
    /**
     * Convert RGB color
     * @return {ColorRGB}
     */
    toRGB(): ColorRGB;
    /**
     * Create CIE color from HSV
     * @return {ColorXY}
     */
    toXY(): ColorXY;
    /**
     * Create Mireds from HSV
     * @return {number} color temp in mireds
     */
    toMireds(): number;
    /**
     * Returns color with missing properties set to defaults
     * @return {ColorHSV} HSV color
     */
    complete(): ColorHSV;
    /**
     * Returns color after sRGB gamma correction
     * @return {ColorHSV} corrected color in HSV space
     */
    gammaCorrected(): ColorHSV;
    /**
     * Returns HSV color after hue correction
     * @param {Object} meta entity meta object
     * @return {ColorHSV} hue corrected color
     */
    hueCorrected(meta: Object): ColorHSV;
    /**
     * Returns HSV color after gamma and hue corrections
     * @param {Object} meta entity meta object
     * @return {ColorHSV} corrected color in HSV space
     */
    colorCorrected(meta: Object): ColorHSV;
}
export class Color {
    /**
     * Create Color object from converter's value argument
     * @param {Object} value converter value argument
     * @return {Color} Color object
     */
    static fromConverterArg(value: Object): Color;
    /**
     * Create Color object
     * @param {?ColorHSV} hsv ColorHSV instance
     * @param {?ColorRGB} rgb ColorRGB instance
     * @param {?ColorXY} xy ColorXY instance
     */
    constructor(hsv: ColorHSV | null, rgb: ColorRGB | null, xy: ColorXY | null);
    hsv: ColorHSV;
    rgb: ColorRGB;
    xy: ColorXY;
    /**
     * Returns true if color is HSV
     * @return {boolean} is HSV color
     */
    isHSV(): boolean;
    /**
     * Returns true if color is RGB
     * @return {boolean} is RGB color
     */
    isRGB(): boolean;
    /**
     * Returns true if color is XY
     * @return {boolean} is XY color
     */
    isXY(): boolean;
}
export function syncColorState(newState: any, oldState: any, endpoint: any, options: any, logger: any): {
    color_mode: any;
    color: any;
    color_temp: any;
};
//# sourceMappingURL=color.d.ts.map