/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import stripAnsi from 'strip-ansi';
import { stripVTControlCharacters } from 'node:util';
/**
 * Calculates the maximum width of a multi-line ASCII art string.
 * @param asciiArt The ASCII art string.
 * @returns The length of the longest line in the ASCII art.
 */
export const getAsciiArtWidth = (asciiArt) => {
    if (!asciiArt) {
        return 0;
    }
    const lines = asciiArt.split('\n');
    return Math.max(...lines.map((line) => line.length));
};
/*
 * -------------------------------------------------------------------------
 *  Unicode‑aware helpers (work at the code‑point level rather than UTF‑16
 *  code units so that surrogate‑pair emoji count as one "column".)
 * ---------------------------------------------------------------------- */
export function toCodePoints(str) {
    // [...str] or Array.from both iterate by UTF‑32 code point, handling
    // surrogate pairs correctly.
    return Array.from(str);
}
export function cpLen(str) {
    return toCodePoints(str).length;
}
export function cpSlice(str, start, end) {
    // Slice by code‑point indices and re‑join.
    const arr = toCodePoints(str).slice(start, end);
    return arr.join('');
}
/**
 * Strip characters that can break terminal rendering.
 *
 * Uses Node.js built-in stripVTControlCharacters to handle VT sequences,
 * then filters remaining control characters that can disrupt display.
 *
 * Characters stripped:
 * - ANSI escape sequences (via strip-ansi)
 * - VT control sequences (via Node.js util.stripVTControlCharacters)
 * - C0 control chars (0x00-0x1F) except CR/LF which are handled elsewhere
 * - C1 control chars (0x80-0x9F) that can cause display issues
 *
 * Characters preserved:
 * - All printable Unicode including emojis
 * - DEL (0x7F) - handled functionally by applyOperations, not a display issue
 * - CR/LF (0x0D/0x0A) - needed for line breaks
 */
export function stripUnsafeCharacters(str) {
    const strippedAnsi = stripAnsi(str);
    const strippedVT = stripVTControlCharacters(strippedAnsi);
    return toCodePoints(strippedVT)
        .filter((char) => {
        const code = char.codePointAt(0);
        if (code === undefined)
            return false;
        // Preserve CR/LF for line handling
        if (code === 0x0a || code === 0x0d)
            return true;
        // Remove C0 control chars (except CR/LF) that can break display
        // Examples: BELL(0x07) makes noise, BS(0x08) moves cursor, VT(0x0B), FF(0x0C)
        if (code >= 0x00 && code <= 0x1f)
            return false;
        // Remove C1 control chars (0x80-0x9f) - legacy 8-bit control codes
        if (code >= 0x80 && code <= 0x9f)
            return false;
        // Preserve DEL (0x7f) - it's handled functionally by applyOperations as backspace
        // and doesn't cause rendering issues when displayed
        // Preserve all other characters including Unicode/emojis
        return true;
    })
        .join('');
}
//# sourceMappingURL=textUtils.js.map