/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { AuthType } from '@qwen-code/qwen-code-core';
import { loadEnvironment } from './settings.js';
export const validateAuthMethod = (authMethod) => {
    loadEnvironment();
    if (authMethod === AuthType.LOGIN_WITH_GOOGLE ||
        authMethod === AuthType.CLOUD_SHELL) {
        return null;
    }
    if (authMethod === AuthType.USE_GEMINI) {
        if (!process.env['GEMINI_API_KEY']) {
            return 'GEMINI_API_KEY environment variable not found. Add that to your environment and try again (no reload needed if using .env)!';
        }
        return null;
    }
    if (authMethod === AuthType.USE_VERTEX_AI) {
        const hasVertexProjectLocationConfig = !!process.env['GOOGLE_CLOUD_PROJECT'] &&
            !!process.env['GOOGLE_CLOUD_LOCATION'];
        const hasGoogleApiKey = !!process.env['GOOGLE_API_KEY'];
        if (!hasVertexProjectLocationConfig && !hasGoogleApiKey) {
            return ('When using Vertex AI, you must specify either:\n' +
                '• GOOGLE_CLOUD_PROJECT and GOOGLE_CLOUD_LOCATION environment variables.\n' +
                '• GOOGLE_API_KEY environment variable (if using express mode).\n' +
                'Update your environment and try again (no reload needed if using .env)!');
        }
        return null;
    }
    if (authMethod === AuthType.USE_OPENAI) {
        if (!process.env['OPENAI_API_KEY']) {
            return 'OPENAI_API_KEY environment variable not found. You can enter it interactively or add it to your .env file.';
        }
        return null;
    }
    if (authMethod === AuthType.QWEN_OAUTH) {
        // Qwen OAuth doesn't require any environment variables for basic setup
        // The OAuth flow will handle authentication
        return null;
    }
    return 'Invalid auth method selected.';
};
export const setOpenAIApiKey = (apiKey) => {
    process.env['OPENAI_API_KEY'] = apiKey;
};
export const setOpenAIBaseUrl = (baseUrl) => {
    process.env['OPENAI_BASE_URL'] = baseUrl;
};
export const setOpenAIModel = (model) => {
    process.env['OPENAI_MODEL'] = model;
};
//# sourceMappingURL=auth.js.map