﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/glue/GlueRequest.h>
#include <aws/glue/Glue_EXPORTS.h>
#include <aws/glue/model/UserDefinedFunctionInput.h>

#include <utility>

namespace Aws {
namespace Glue {
namespace Model {

/**
 */
class CreateUserDefinedFunctionRequest : public GlueRequest {
 public:
  AWS_GLUE_API CreateUserDefinedFunctionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateUserDefinedFunction"; }

  AWS_GLUE_API Aws::String SerializePayload() const override;

  AWS_GLUE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the Data Catalog in which to create the function. If none is
   * provided, the Amazon Web Services account ID is used by default.</p>
   */
  inline const Aws::String& GetCatalogId() const { return m_catalogId; }
  inline bool CatalogIdHasBeenSet() const { return m_catalogIdHasBeenSet; }
  template <typename CatalogIdT = Aws::String>
  void SetCatalogId(CatalogIdT&& value) {
    m_catalogIdHasBeenSet = true;
    m_catalogId = std::forward<CatalogIdT>(value);
  }
  template <typename CatalogIdT = Aws::String>
  CreateUserDefinedFunctionRequest& WithCatalogId(CatalogIdT&& value) {
    SetCatalogId(std::forward<CatalogIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the catalog database in which to create the function.</p>
   */
  inline const Aws::String& GetDatabaseName() const { return m_databaseName; }
  inline bool DatabaseNameHasBeenSet() const { return m_databaseNameHasBeenSet; }
  template <typename DatabaseNameT = Aws::String>
  void SetDatabaseName(DatabaseNameT&& value) {
    m_databaseNameHasBeenSet = true;
    m_databaseName = std::forward<DatabaseNameT>(value);
  }
  template <typename DatabaseNameT = Aws::String>
  CreateUserDefinedFunctionRequest& WithDatabaseName(DatabaseNameT&& value) {
    SetDatabaseName(std::forward<DatabaseNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A <code>FunctionInput</code> object that defines the function to create in
   * the Data Catalog.</p>
   */
  inline const UserDefinedFunctionInput& GetFunctionInput() const { return m_functionInput; }
  inline bool FunctionInputHasBeenSet() const { return m_functionInputHasBeenSet; }
  template <typename FunctionInputT = UserDefinedFunctionInput>
  void SetFunctionInput(FunctionInputT&& value) {
    m_functionInputHasBeenSet = true;
    m_functionInput = std::forward<FunctionInputT>(value);
  }
  template <typename FunctionInputT = UserDefinedFunctionInput>
  CreateUserDefinedFunctionRequest& WithFunctionInput(FunctionInputT&& value) {
    SetFunctionInput(std::forward<FunctionInputT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_catalogId;

  Aws::String m_databaseName;

  UserDefinedFunctionInput m_functionInput;
  bool m_catalogIdHasBeenSet = false;
  bool m_databaseNameHasBeenSet = false;
  bool m_functionInputHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glue
}  // namespace Aws
