﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticloadbalancingv2/ElasticLoadBalancingv2Request.h>
#include <aws/elasticloadbalancingv2/ElasticLoadBalancingv2_EXPORTS.h>
#include <aws/elasticloadbalancingv2/model/Certificate.h>

#include <utility>

namespace Aws {
namespace ElasticLoadBalancingv2 {
namespace Model {

/**
 */
class RemoveListenerCertificatesRequest : public ElasticLoadBalancingv2Request {
 public:
  AWS_ELASTICLOADBALANCINGV2_API RemoveListenerCertificatesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RemoveListenerCertificates"; }

  AWS_ELASTICLOADBALANCINGV2_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICLOADBALANCINGV2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the listener.</p>
   */
  inline const Aws::String& GetListenerArn() const { return m_listenerArn; }
  inline bool ListenerArnHasBeenSet() const { return m_listenerArnHasBeenSet; }
  template <typename ListenerArnT = Aws::String>
  void SetListenerArn(ListenerArnT&& value) {
    m_listenerArnHasBeenSet = true;
    m_listenerArn = std::forward<ListenerArnT>(value);
  }
  template <typename ListenerArnT = Aws::String>
  RemoveListenerCertificatesRequest& WithListenerArn(ListenerArnT&& value) {
    SetListenerArn(std::forward<ListenerArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The certificate to remove. You can specify one certificate per call. Set
   * <code>CertificateArn</code> to the certificate ARN but do not set
   * <code>IsDefault</code>.</p>
   */
  inline const Aws::Vector<Certificate>& GetCertificates() const { return m_certificates; }
  inline bool CertificatesHasBeenSet() const { return m_certificatesHasBeenSet; }
  template <typename CertificatesT = Aws::Vector<Certificate>>
  void SetCertificates(CertificatesT&& value) {
    m_certificatesHasBeenSet = true;
    m_certificates = std::forward<CertificatesT>(value);
  }
  template <typename CertificatesT = Aws::Vector<Certificate>>
  RemoveListenerCertificatesRequest& WithCertificates(CertificatesT&& value) {
    SetCertificates(std::forward<CertificatesT>(value));
    return *this;
  }
  template <typename CertificatesT = Certificate>
  RemoveListenerCertificatesRequest& AddCertificates(CertificatesT&& value) {
    m_certificatesHasBeenSet = true;
    m_certificates.emplace_back(std::forward<CertificatesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_listenerArn;

  Aws::Vector<Certificate> m_certificates;
  bool m_listenerArnHasBeenSet = false;
  bool m_certificatesHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticLoadBalancingv2
}  // namespace Aws
