/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: LEFDEF.java
 * Input/output tool: LEF and DEF helpers
 * Written by Steven M. Rubin, Sun Microsystems.
 *
 * Copyright (c) 2004, Oracle and/or its affiliates. All rights reserved.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.sun.electric.tool.io.input;

import com.sun.electric.database.EditingPreferences;
import com.sun.electric.database.prototype.NodeProto;
import com.sun.electric.database.prototype.PortProto;
import com.sun.electric.database.variable.Variable;
import com.sun.electric.technology.ArcProto;
import com.sun.electric.technology.Layer;
import com.sun.electric.technology.PrimitiveNode;
import com.sun.electric.technology.Technology;
import com.sun.electric.technology.Technology.ArcLayer;
import com.sun.electric.technology.Technology.NodeLayer;
import com.sun.electric.util.TextUtils;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * This class defines supporting structures and methods for reading of LEF and DEF files.
 */
public class LEFDEF extends Input<Object>
{
	protected static final boolean PLACELEFGEOMETRY = true;
	protected static final boolean PLACELEFEXPORTS = true;

	protected static Map<String,ViaDef> viaDefsFromLEF;
	protected static Map<ArcProto,Double> widthsFromLEF;
	protected static Map<String,GetLayerInformation> knownLayers;
	protected static final double OVERALLSCALE = 1;

	protected static Variable.Key prXkey = Variable.newKey("ATTR_LEFwidth");
	protected static Variable.Key prYkey = Variable.newKey("ATTR_LEFheight");

	protected Technology curTech;
	private boolean viaDigitsCombine;

	/**
	 * Class to define Via information for LEF and DEF reading.
	 */
	protected static class ViaDef
	{
		protected String    viaName;
		protected NodeProto via;
//		protected ArcProto  lay1, lay2;
		protected GetLayerInformation gLay1, gLay2;
		protected double    sX, sY;

		public ViaDef(String name, NodeProto np)
		{
			viaName = name;
			sX = sY = 0;
			via = np;
//			lay1 = lay2 = null;
			gLay1 = gLay2 = null;
		}
	};

    LEFDEF(EditingPreferences ep) {
        super(ep);
    }
    
	protected void initializeLEFDEF(Technology tech)
	{
		curTech = tech;
		viaDigitsCombine = false;
		knownLayers = new HashMap<String,GetLayerInformation>();
    	viaDefsFromLEF = new HashMap<String,ViaDef>();
	}

	/**
	 * Method to find a layer from its name.
	 * Uses a map of known layers, and analyzes the name if none is found.
	 * @param name the layer name.
	 * @return the layer information object.
	 */
	protected GetLayerInformation getLayerInformation(String name)
	{
		GetLayerInformation li = knownLayers.get(name);
		if (li != null) return li;

		li = new GetLayerInformation(name);
		knownLayers.put(name, li);
		return li;
	}

	/**
	 * Class to define layer information for LEF and DEF reading.
	 */
	protected class GetLayerInformation
	{
		String name;
		NodeProto pin;
		NodeProto pure;
		ArcProto arc;
		ArcProto.Function arcFun;
		Layer.Function layerFun;
		ArcProto viaArc1, viaArc2;

		public boolean equals(GetLayerInformation other)
		{
			if (layerFun == other.layerFun && arcFun == other.arcFun) return true;
			return false;
		}

		private NodeProto getPureLayerNode()
		{
			// find the pure layer node with this function
			for(Iterator<Layer> it = curTech.getLayers(); it.hasNext(); )
			{
				Layer lay = it.next();
				if (lay.getFunction() == layerFun)
				{
					return lay.getPureLayerNode();
				}
			}
			return null;
		}

		GetLayerInformation(String name)
		{
			initialize(name, null);
		}

		GetLayerInformation(String name, String type)
		{
			initialize(name, type);
		}

		private void initialize(String name, String type)
		{
			// initialize
			this.name = name;
			pin = null;
			pure = null;
			arc = null;
			arcFun = ArcProto.Function.UNKNOWN;
			layerFun = Layer.Function.UNKNOWN;
			viaArc1 = viaArc2 = null;
			int colonPos = name.indexOf(':');
			if (colonPos >= 0)
			{
				// given full TECH:LAYER name
				Technology tech = Technology.findTechnology(name.substring(0, colonPos));
				if (tech != null)
				{
					String layerName = name.substring(colonPos+1);
					Layer lay = tech.findLayer(layerName);
					if (lay != null)
					{
						layerFun = lay.getFunction();
						pure = lay.getPureLayerNode();
						for(Iterator<ArcProto> it = tech.getArcs(); it.hasNext(); )
						{
							ArcProto ap = it.next();
							ArcLayer[] parts = ap.getArcLayers();
							for(int i=0; i<parts.length; i++)
							{
								if (parts[i].getLayer() == lay)
								{
									arc = ap;
									arcFun = arc.getFunction();
									pin = arc.findPinProto();
									break;
								}
							}
							if (arc != null) break;
						}
						return;
					}
				}
			}

			// analyze based on generic names
			name = name.toUpperCase();

			// first handle known layer names
			if (name.startsWith("POLY"))
			{
				setupPolyLayer(name.substring(4));
				return;
			}
			if (name.startsWith("PDIF"))
			{
				arcFun = ArcProto.Function.DIFFP;
				layerFun = Layer.Function.DIFFP;
				pure = getPureLayerNode();
				return;
			}
			if (name.startsWith("NDIF"))
			{
				arcFun = ArcProto.Function.DIFFN;
				layerFun = Layer.Function.DIFFN;
				pure = getPureLayerNode();
				return;
			}
			if (name.startsWith("PWEL"))
			{
				layerFun = Layer.Function.WELLP;
				pure = getPureLayerNode();
				return;
			}
			if (name.startsWith("NWEL"))
			{
				layerFun = Layer.Function.WELLN;
				pure = getPureLayerNode();
				return;
			}
			if (name.equals("DIFF"))
			{
				arcFun = ArcProto.Function.DIFF;
				layerFun = Layer.Function.DIFF;
				pure = getPureLayerNode();
				return;
			}
			if (name.equals("CONT") || name.equals("CON") || name.equals("CO"))
			{
				layerFun = Layer.Function.CONTACT1;
				pure = getPureLayerNode();
				return;
			}

			// handle via layers
			int j = 0;
			if (name.startsWith("VIA")) j = 3; else
				if (name.startsWith("V")) j = 1;
			if (j != 0)
			{
				if (setupViaLayer(name.substring(j))) return;
			}

			// handle metal layers
			j = 0;
			if (name.startsWith("METAL")) j = 5; else
				if (name.startsWith("MET")) j = 3; else
					if (name.startsWith("M")) j = 1;
			if (j != 0)
			{
				setupMetalLayer(name.substring(j));
				return;
			}

			// if type is given, use it
			if (type != null)
			{
				if (type.equalsIgnoreCase("masterslice"))
				{
					// masterslice layers are typically polysilicon
					j = 0;
					name = name.toUpperCase();
					if (name.startsWith("POLY")) j = 4; else
						if (name.startsWith("P")) j = 1;
					setupPolyLayer(name.substring(j));
					return;
				}
				if (type.equalsIgnoreCase("cut"))
				{
					j = 0;
					name = name.toUpperCase();
					if (name.startsWith("VIA")) j = 3; else
						if (name.startsWith("V")) j = 1;
					if (setupViaLayer(name.substring(j))) return;
				}
				if (type.equalsIgnoreCase("routing"))
				{
					j = 0;
					if (name.startsWith("METAL")) j = 5; else
						if (name.startsWith("MET")) j = 3; else
							if (name.startsWith("M")) j = 1;
					name = name.substring(j);
					while (name.length() > 0 && !Character.isDigit(name.charAt(0)))
						name = name.substring(1);
					setupMetalLayer(name);
					return;
				}
			}

			// look for a layer that starts with the name
			for(Iterator<Layer> it = curTech.getLayers(); it.hasNext(); )
			{
				Layer lay = it.next();
				if (lay.getName().startsWith(name))
				{
					layerFun = lay.getFunction();
					assignPureNodeBasedOnLay(lay);
					/*
					for(Iterator<PrimitiveNode> pIt = curTech.getNodes(); pIt.hasNext(); )
					{
						PrimitiveNode pn = pIt.next();
						if (pn.getFunction() != PrimitiveNode.Function.NODE) continue;
						NodeLayer[] layersOnNode = pn.getNodeLayers();
						if (layersOnNode[0].getLayer() == lay)
						{
							pure = pn;
							break;
						}
					}
					*/
					return;
				}
			}

			// special cases
			if (name.indexOf("OVERLAP") >= 0)
			{
				for(Iterator<Layer> it = curTech.getLayers(); it.hasNext(); )
				{
					Layer lay = it.next();
					if (lay.getName().toLowerCase().indexOf("prbound") >= 0)
					{
						layerFun = lay.getFunction();
						assignPureNodeBasedOnLay(lay);
						return;
					}
				}
				
			}
			System.out.println("Error: didn't find in initialize a layer to match '" + name + "'");
		}

		private void assignPureNodeBasedOnLay(Layer lay)
		{
			for(Iterator<PrimitiveNode> pIt = curTech.getNodes(); pIt.hasNext(); )
			{
				PrimitiveNode pn = pIt.next();
				if (pn.getFunction() != PrimitiveNode.Function.NODE) continue;
				NodeLayer[] layersOnNode = pn.getNodeLayers();
				if (layersOnNode[0].getLayer() == lay)
				{
					pure = pn;
					return;
				}
			}
			System.out.println("Error: didn't find in initialize a pure node to match layer '" + lay.getName() + "'");
		}
		
		private void setupMetalLayer(String name)
		{
			int layNum = TextUtils.atoi(name);
			arcFun = ArcProto.Function.getMetal(layNum);
			layerFun = Layer.Function.getMetal(layNum);
			if (arcFun == null || layerFun == null) return;

			// find the arc with this function
			for(Iterator<ArcProto> it = curTech.getArcs(); it.hasNext(); )
			{
				ArcProto ap = it.next();
				if (ap.getFunction() == arcFun)
				{
					arc = ap;
					pin = ap.findPinProto();
					break;
				}
			}

			// find the pure layer node with this function
			pure = getPureLayerNode();
		}

		private void setupPolyLayer(String name)
		{			
			int layNum = TextUtils.atoi(name);
			if (layNum == 0) layNum = 1;
			arcFun = ArcProto.Function.getPoly(layNum);
			layerFun = Layer.Function.getPoly(layNum);
			if (arcFun == null || layerFun == null) return;

			// find the arc with this function
			for(Iterator<ArcProto> it = curTech.getArcs(); it.hasNext(); )
			{
				ArcProto ap = it.next();
				if (ap.getFunction() == arcFun)
				{
					arc = ap;
					pin = ap.findPinProto();
					break;
				}
			}

			// find the pure layer node with this function
			pure = getPureLayerNode();
		}

		private boolean setupViaLayer(String name)
		{
			// find the two layer functions
			ArcProto.Function aFunc1 = ArcProto.Function.UNKNOWN;
			ArcProto.Function aFunc2 = ArcProto.Function.UNKNOWN;
			if (name.length() <= 0)
			{
				aFunc1 = ArcProto.Function.METAL1;
				aFunc2 = ArcProto.Function.METAL2;
			} else if (name.length() <= 1)
			{
				int level = name.charAt(0) - '0';
				if (level < 0 || level > 9) return false;
				if (level == 0) aFunc1 = ArcProto.Function.getPoly(1); else
					aFunc1 = ArcProto.Function.getMetal(level);
				aFunc2 = ArcProto.Function.getMetal(level + 1);
			} else
			{
				int level1 = name.charAt(0) - '0';
				int level2 = name.charAt(1) - '0';
				if (level1 < 0 || level1 > 9) return false;
				if (level2 < 0 || level2 > 9) return false;
				if (!viaDigitsCombine && level2 <= level1) viaDigitsCombine = true;
				if (viaDigitsCombine)
				{
					level1 = level1*10 + level2;
					level2 = level1 + 1;
				}
				if (level1 == 0) aFunc1 = ArcProto.Function.getPoly(1); else
					aFunc1 = ArcProto.Function.getMetal(level1);
				aFunc2 = ArcProto.Function.getMetal(level2);
			}

			// find the arcprotos that embody these layers
			for(Iterator<ArcProto> it = curTech.getArcs(); it.hasNext(); )
			{
				ArcProto apTry = it.next();
				if (apTry.getFunction() == aFunc1) viaArc1 = apTry;
				if (apTry.getFunction() == aFunc2) viaArc2 = apTry;
			}
			if (viaArc1 == null || viaArc2 == null) return false;

			// find the via that connects these two arcs
			for(Iterator<PrimitiveNode> it = curTech.getNodes(); it.hasNext(); )
			{
				PrimitiveNode np = it.next();
				// must have just one port
				if (np.getNumPorts() != 1) continue;

				// port must connect to both arcs
				PortProto pp = np.getPort(0);
				boolean ap1Found = pp.connectsTo(viaArc1);
				boolean ap2Found = pp.connectsTo(viaArc2);
				if (ap1Found && ap2Found) { pin = np;   break; }
			}

			// find the pure layer node that is the via contact
			if (pin != null)
			{
				// find the layer on this node that is of type "contact"
				PrimitiveNode pNp = (PrimitiveNode)pin;
				Technology.NodeLayer [] nl = pNp.getNodeLayers();
				Layer viaLayer = null;
				for(int i=0; i<nl.length; i++)
				{
					Technology.NodeLayer nLay = nl[i];
					Layer lay = nLay.getLayer();
					Layer.Function fun = lay.getFunction();
					if (fun.isContact()) { viaLayer = lay;   layerFun = fun;   break; }
				}
				if (viaLayer == null) return false;
				pure = viaLayer.getPureLayerNode();
			}
			return true;
		}
	}

}
