package org.mandas.docker.client.builder;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link ProxyConfiguration}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableProxyConfiguration.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableProxyConfiguration implements ProxyConfiguration {
  private final String host;
  private final String port;
  private final @Nullable String username;
  private final @Nullable String password;

  private ImmutableProxyConfiguration(
      String host,
      String port,
      @Nullable String username,
      @Nullable String password) {
    this.host = host;
    this.port = port;
    this.username = username;
    this.password = password;
  }

  /**
   * @return The value of the {@code host} attribute
   */
  @Override
  public String host() {
    return host;
  }

  /**
   * @return The value of the {@code port} attribute
   */
  @Override
  public String port() {
    return port;
  }

  /**
   * @return The value of the {@code username} attribute
   */
  @Override
  public @Nullable String username() {
    return username;
  }

  /**
   * @return The value of the {@code password} attribute
   */
  @Override
  public @Nullable String password() {
    return password;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProxyConfiguration#host() host} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for host
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableProxyConfiguration withHost(String value) {
    String newValue = Objects.requireNonNull(value, "host");
    if (this.host.equals(newValue)) return this;
    return new ImmutableProxyConfiguration(newValue, this.port, this.username, this.password);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProxyConfiguration#port() port} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for port
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableProxyConfiguration withPort(String value) {
    String newValue = Objects.requireNonNull(value, "port");
    if (this.port.equals(newValue)) return this;
    return new ImmutableProxyConfiguration(this.host, newValue, this.username, this.password);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProxyConfiguration#username() username} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for username (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableProxyConfiguration withUsername(@Nullable String value) {
    if (Objects.equals(this.username, value)) return this;
    return new ImmutableProxyConfiguration(this.host, this.port, value, this.password);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ProxyConfiguration#password() password} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for password (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableProxyConfiguration withPassword(@Nullable String value) {
    if (Objects.equals(this.password, value)) return this;
    return new ImmutableProxyConfiguration(this.host, this.port, this.username, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableProxyConfiguration} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableProxyConfiguration
        && equalsByValue((ImmutableProxyConfiguration) another);
  }

  private boolean equalsByValue(ImmutableProxyConfiguration another) {
    return host.equals(another.host)
        && port.equals(another.port)
        && Objects.equals(username, another.username)
        && Objects.equals(password, another.password);
  }

  /**
   * Computes a hash code from attributes: {@code host}, {@code port}, {@code username}, {@code password}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + host.hashCode();
    h += (h << 5) + port.hashCode();
    h += (h << 5) + Objects.hashCode(username);
    h += (h << 5) + Objects.hashCode(password);
    return h;
  }

  /**
   * Prints the immutable value {@code ProxyConfiguration} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ProxyConfiguration{"
        + "host=" + host
        + ", port=" + port
        + ", username=" + username
        + ", password=" + password
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ProxyConfiguration} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ProxyConfiguration instance
   */
  public static ImmutableProxyConfiguration copyOf(ProxyConfiguration instance) {
    if (instance instanceof ImmutableProxyConfiguration) {
      return (ImmutableProxyConfiguration) instance;
    }
    return ImmutableProxyConfiguration.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableProxyConfiguration ImmutableProxyConfiguration}.
   * <pre>
   * ImmutableProxyConfiguration.builder()
   *    .host(String) // required {@link ProxyConfiguration#host() host}
   *    .port(String) // required {@link ProxyConfiguration#port() port}
   *    .username(String | null) // nullable {@link ProxyConfiguration#username() username}
   *    .password(String | null) // nullable {@link ProxyConfiguration#password() password}
   *    .build();
   * </pre>
   * @return A new ImmutableProxyConfiguration builder
   */
  public static ImmutableProxyConfiguration.Builder builder() {
    return new ImmutableProxyConfiguration.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableProxyConfiguration ImmutableProxyConfiguration}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements ProxyConfiguration.Builder {
    private static final long INIT_BIT_HOST = 0x1L;
    private static final long INIT_BIT_PORT = 0x2L;
    private long initBits = 0x3L;

    private String host;
    private String port;
    private String username;
    private String password;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ProxyConfiguration} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ProxyConfiguration instance) {
      Objects.requireNonNull(instance, "instance");
      this.host(instance.host());
      this.port(instance.port());
      @Nullable String usernameValue = instance.username();
      if (usernameValue != null) {
        username(usernameValue);
      }
      @Nullable String passwordValue = instance.password();
      if (passwordValue != null) {
        password(passwordValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ProxyConfiguration#host() host} attribute.
     * @param host The value for host 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder host(String host) {
      this.host = Objects.requireNonNull(host, "host");
      initBits &= ~INIT_BIT_HOST;
      return this;
    }

    /**
     * Initializes the value for the {@link ProxyConfiguration#port() port} attribute.
     * @param port The value for port 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder port(String port) {
      this.port = Objects.requireNonNull(port, "port");
      initBits &= ~INIT_BIT_PORT;
      return this;
    }

    /**
     * Initializes the value for the {@link ProxyConfiguration#username() username} attribute.
     * @param username The value for username (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder username(@Nullable String username) {
      this.username = username;
      return this;
    }

    /**
     * Initializes the value for the {@link ProxyConfiguration#password() password} attribute.
     * @param password The value for password (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder password(@Nullable String password) {
      this.password = password;
      return this;
    }

    /**
     * Builds a new {@link ImmutableProxyConfiguration ImmutableProxyConfiguration}.
     * @return An immutable instance of ProxyConfiguration
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableProxyConfiguration build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableProxyConfiguration(host, port, username, password);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_HOST) != 0) attributes.add("host");
      if ((initBits & INIT_BIT_PORT) != 0) attributes.add("port");
      return "Cannot build ProxyConfiguration, some of required attributes are not set " + attributes;
    }
  }
}
