package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Driver}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableDriver.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableDriver implements Driver {
  private final @Nullable String name;
  private final Map<String, String> options;

  private ImmutableDriver(
      @Nullable String name,
      Map<String, String> options) {
    this.name = name;
    this.options = options;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @JsonProperty("Name")
  @Override
  public @Nullable String name() {
    return name;
  }

  /**
   * @return The value of the {@code options} attribute
   */
  @JsonProperty("Options")
  @Override
  public Map<String, String> options() {
    return options;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Driver#name() name} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDriver withName(@Nullable String value) {
    if (Objects.equals(this.name, value)) return this;
    return new ImmutableDriver(value, this.options);
  }

  /**
   * Copy the current immutable object by replacing the {@link Driver#options() options} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to be added to the options map
   * @return A modified copy of {@code this} object
   */
  public final ImmutableDriver withOptions(Map<String, ? extends String> entries) {
    if (this.options == entries) return this;
    Map<String, String> newValue = createUnmodifiableMap(true, false, entries);
    return new ImmutableDriver(this.name, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableDriver} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableDriver
        && equalTo(0, (ImmutableDriver) another);
  }

  private boolean equalTo(int synthetic, ImmutableDriver another) {
    return Objects.equals(name, another.name)
        && options.equals(another.options);
  }

  /**
   * Computes a hash code from attributes: {@code name}, {@code options}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(name);
    h += (h << 5) + options.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Driver} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Driver{"
        + "name=" + name
        + ", options=" + options
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Driver} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Driver instance
   */
  public static ImmutableDriver copyOf(Driver instance) {
    if (instance instanceof ImmutableDriver) {
      return (ImmutableDriver) instance;
    }
    return ImmutableDriver.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableDriver ImmutableDriver}.
   * <pre>
   * ImmutableDriver.builder()
   *    .name(String | null) // nullable {@link Driver#name() name}
   *    .addOption|putAllOptions(String =&gt; String) // {@link Driver#options() options} mappings
   *    .build();
   * </pre>
   * @return A new ImmutableDriver builder
   */
  public static ImmutableDriver.Builder builder() {
    return new ImmutableDriver.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableDriver ImmutableDriver}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements Driver.Builder {
    private String name;
    private Map<String, String> options = new LinkedHashMap<String, String>();

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Driver} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Driver instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable String nameValue = instance.name();
      if (nameValue != null) {
        name(nameValue);
      }
      putAllOptions(instance.options());
      return this;
    }

    /**
     * Initializes the value for the {@link Driver#name() name} attribute.
     * @param name The value for name (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Name")
    public final Builder name(@Nullable String name) {
      this.name = name;
      return this;
    }

    /**
     * Put one entry to the {@link Driver#options() options} map.
     * @param key The key in the options map
     * @param value The associated value in the options map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addOption(String key, String value) {
      this.options.put(
          Objects.requireNonNull(key, "options key"),
          Objects.requireNonNull(value, value == null ? "options value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link Driver#options() options} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addOption(Map.Entry<String, ? extends String> entry) {
      String k = entry.getKey();
      String v = entry.getValue();
      this.options.put(
          Objects.requireNonNull(k, "options key"),
          Objects.requireNonNull(v, v == null ? "options value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link Driver#options() options} map. Nulls are not permitted
     * @param entries The entries that will be added to the options map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Options")
    public final Builder options(Map<String, ? extends String> entries) {
      this.options.clear();
      return putAllOptions(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link Driver#options() options} map. Nulls are not permitted
     * @param entries The entries that will be added to the options map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllOptions(Map<String, ? extends String> entries) {
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.options.put(
            Objects.requireNonNull(k, "options key"),
            Objects.requireNonNull(v, v == null ? "options value for key: " + k : null));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableDriver ImmutableDriver}.
     * @return An immutable instance of Driver
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableDriver build() {
      return new ImmutableDriver(name, createUnmodifiableMap(false, false, options));
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
