from __future__ import annotations

import warnings
from typing import TYPE_CHECKING, Any, NamedTuple

from ..actions import (
    ActionSort,
    ActionsPageResult,
    ActionStatus,
    BoundAction,
    ResourceActionsClient,
)
from ..actions.client import ResourceClientBaseActionsMixin
from ..core import BoundModelBase, Meta, ResourceClientBase
from .domain import CreatePrimaryIPResponse, PrimaryIP

if TYPE_CHECKING:
    from .._client import Client
    from ..datacenters import BoundDatacenter, Datacenter
    from ..locations import BoundLocation, Location


__all__ = [
    "BoundPrimaryIP",
    "PrimaryIPsPageResult",
    "PrimaryIPsClient",
]


class BoundPrimaryIP(BoundModelBase[PrimaryIP], PrimaryIP):
    _client: PrimaryIPsClient

    model = PrimaryIP

    def __init__(
        self,
        client: PrimaryIPsClient,
        data: dict[str, Any],
        complete: bool = True,
    ):
        # pylint: disable=import-outside-toplevel
        from ..datacenters import BoundDatacenter
        from ..locations import BoundLocation

        raw = data.get("datacenter", {})
        if raw:
            data["datacenter"] = BoundDatacenter(client._parent.datacenters, raw)

        raw = data.get("location", {})
        if raw:
            data["location"] = BoundLocation(client._parent.locations, raw)

        super().__init__(client, data, complete)

    def get_actions_list(
        self,
        status: list[ActionStatus] | None = None,
        sort: list[ActionSort] | None = None,
        page: int | None = None,
        per_page: int | None = None,
    ) -> ActionsPageResult:
        """
        Returns a paginated list of Actions for a Primary IP.

        :param status: Filter the Actions by status.
        :param sort: Sort Actions by field and direction.
        :param page: Page number to get.
        :param per_page: Maximum number of Actions returned per page.
        """
        return self._client.get_actions_list(
            self,
            status=status,
            sort=sort,
            page=page,
            per_page=per_page,
        )

    def get_actions(
        self,
        status: list[ActionStatus] | None = None,
        sort: list[ActionSort] | None = None,
    ) -> list[BoundAction]:
        """
        Returns all Actions for a Primary IP.

        :param status: Filter the Actions by status.
        :param sort: Sort Actions by field and direction.
        """
        return self._client.get_actions(
            self,
            status=status,
            sort=sort,
        )

    def update(
        self,
        auto_delete: bool | None = None,
        labels: dict[str, str] | None = None,
        name: str | None = None,
    ) -> BoundPrimaryIP:
        """Updates the description or labels of a Primary IP.

        :param auto_delete: bool (optional)
               Auto delete IP when assignee gets deleted
        :param labels: Dict[str, str] (optional)
               User-defined labels (key-value pairs)
        :param name: str (optional)
               New Name to set
        :return: :class:`BoundPrimaryIP <hcloud.primary_ips.client.BoundPrimaryIP>`
        """
        return self._client.update(
            self,
            auto_delete=auto_delete,
            labels=labels,
            name=name,
        )

    def delete(self) -> bool:
        """Deletes a Primary IP. If it is currently assigned to a server it will automatically get unassigned.

        :return: boolean
        """
        return self._client.delete(self)

    def change_protection(self, delete: bool | None = None) -> BoundAction:
        """Changes the protection configuration of the Primary IP.

        :param delete: boolean
               If true, prevents the Primary IP from being deleted
        :return: :class:`BoundAction <hcloud.actions.client.BoundAction>`
        """
        return self._client.change_protection(self, delete=delete)

    def assign(self, assignee_id: int, assignee_type: str) -> BoundAction:
        """Assigns a Primary IP to a assignee.

        :param assignee_id: int`
               Id of an assignee the Primary IP shall be assigned to
        :param assignee_type: string`
               Assignee type (e.g server) the Primary IP shall be assigned to
        :return: :class:`BoundAction <hcloud.actions.client.BoundAction>`
        """
        return self._client.assign(
            self, assignee_id=assignee_id, assignee_type=assignee_type
        )

    def unassign(self) -> BoundAction:
        """Unassigns a Primary IP, resulting in it being unreachable. You may assign it to a server again at a later time.

        :return: :class:`BoundAction <hcloud.actions.client.BoundAction>`
        """
        return self._client.unassign(self)

    def change_dns_ptr(self, ip: str, dns_ptr: str) -> BoundAction:
        """Changes the hostname that will appear when getting the hostname belonging to this Primary IP.

        :param ip: str
               The IP address for which to set the reverse DNS entry
        :param dns_ptr: str
               Hostname to set as a reverse DNS PTR entry, will reset to original default value if `None`
        :return: :class:`BoundAction <hcloud.actions.client.BoundAction>`
        """
        return self._client.change_dns_ptr(self, ip=ip, dns_ptr=dns_ptr)


class PrimaryIPsPageResult(NamedTuple):
    primary_ips: list[BoundPrimaryIP]
    meta: Meta


class PrimaryIPsClient(
    ResourceClientBaseActionsMixin,
    ResourceClientBase,
):
    _base_url = "/primary_ips"

    actions: ResourceActionsClient
    """Primary IPs scoped actions client

    :type: :class:`ResourceActionsClient <hcloud.actions.client.ResourceActionsClient>`
    """

    def __init__(self, client: Client):
        super().__init__(client)
        self.actions = ResourceActionsClient(client, self._base_url)

    def get_actions_list(
        self,
        primary_ip: PrimaryIP | BoundPrimaryIP,
        status: list[ActionStatus] | None = None,
        sort: list[ActionSort] | None = None,
        page: int | None = None,
        per_page: int | None = None,
    ) -> ActionsPageResult:
        """
        Returns a paginated list of Actions for a Primary IP.

        :param primary_ip: Primary IP to get the Actions for.
        :param status: Filter the Actions by status.
        :param sort: Sort Actions by field and direction.
        :param page: Page number to get.
        :param per_page: Maximum number of Actions returned per page.
        """
        return self._get_actions_list(
            f"{self._base_url}/{primary_ip.id}",
            status=status,
            sort=sort,
            page=page,
            per_page=per_page,
        )

    def get_actions(
        self,
        primary_ip: PrimaryIP | BoundPrimaryIP,
        status: list[ActionStatus] | None = None,
        sort: list[ActionSort] | None = None,
    ) -> list[BoundAction]:
        """
        Returns all Actions for a Primary IP.

        :param primary_ip: Primary IP to get the Actions for.
        :param status: Filter the Actions by status.
        :param sort: Sort Actions by field and direction.
        """
        return self._iter_pages(
            self.get_actions_list,
            primary_ip,
            status=status,
            sort=sort,
        )

    def get_by_id(self, id: int) -> BoundPrimaryIP:
        """Returns a specific Primary IP object.

        :param id: int
        :return: :class:`BoundPrimaryIP <hcloud.primary_ips.client.BoundPrimaryIP>`
        """
        response = self._client.request(url=f"{self._base_url}/{id}", method="GET")
        return BoundPrimaryIP(self, response["primary_ip"])

    def get_list(
        self,
        label_selector: str | None = None,
        page: int | None = None,
        per_page: int | None = None,
        name: str | None = None,
        ip: str | None = None,
    ) -> PrimaryIPsPageResult:
        """Get a list of primary ips from this account

        :param label_selector: str (optional)
               Can be used to filter Primary IPs by labels. The response will only contain Primary IPs matching the label selectorable values.
        :param page: int (optional)
               Specifies the page to fetch
        :param per_page: int (optional)
               Specifies how many results are returned by page
        :param name: str (optional)
               Can be used to filter networks by their name.
        :param ip: str (optional)
               Can be used to filter resources by their ip. The response will only contain the resources matching the specified ip.
        :return: (List[:class:`BoundPrimaryIP <hcloud.primary_ips.client.BoundPrimaryIP>`], :class:`Meta <hcloud.core.domain.Meta>`)
        """
        params: dict[str, Any] = {}

        if label_selector is not None:
            params["label_selector"] = label_selector
        if page is not None:
            params["page"] = page
        if per_page is not None:
            params["per_page"] = per_page
        if name is not None:
            params["name"] = name
        if ip is not None:
            params["ip"] = ip

        response = self._client.request(url=self._base_url, method="GET", params=params)
        primary_ips = [
            BoundPrimaryIP(self, primary_ip_data)
            for primary_ip_data in response["primary_ips"]
        ]

        return PrimaryIPsPageResult(primary_ips, Meta.parse_meta(response))

    def get_all(
        self,
        label_selector: str | None = None,
        name: str | None = None,
    ) -> list[BoundPrimaryIP]:
        """Get all primary ips from this account

        :param label_selector: str (optional)
               Can be used to filter Primary IPs by labels. The response will only contain Primary IPs matching the label selector.able values.
        :param name: str (optional)
               Can be used to filter networks by their name.
        :return: List[:class:`BoundPrimaryIP <hcloud.primary_ips.client.BoundPrimaryIP>`]
        """
        return self._iter_pages(self.get_list, label_selector=label_selector, name=name)

    def get_by_name(self, name: str) -> BoundPrimaryIP | None:
        """Get Primary IP by name

        :param name: str
               Used to get Primary IP by name.
        :return: :class:`BoundPrimaryIP <hcloud.primary_ips.client.BoundPrimaryIP>`
        """
        return self._get_first_by(self.get_list, name=name)

    def create(
        self,
        type: str,
        name: str,
        datacenter: Datacenter | BoundDatacenter | None = None,
        location: Location | BoundLocation | None = None,
        assignee_type: str | None = "server",
        assignee_id: int | None = None,
        auto_delete: bool | None = False,
        labels: dict[str, str] | None = None,
    ) -> CreatePrimaryIPResponse:
        """Creates a new Primary IP assigned to a server.

        :param type: str Primary IP type Choices: ipv4, ipv6
        :param name: str
        :param datacenter: Datacenter (optional)
        :param location: Location (optional)
        :param assignee_type: str (optional)
        :param assignee_id: int (optional)
        :param auto_delete: bool (optional)
        :param labels: Dict[str, str] (optional) User-defined labels (key-value pairs)
        :return: :class:`CreatePrimaryIPResponse <hcloud.primary_ips.domain.CreatePrimaryIPResponse>`
        """

        data: dict[str, Any] = {
            "name": name,
            "type": type,
            "assignee_type": assignee_type,
            "auto_delete": auto_delete,
        }
        if datacenter is not None:
            warnings.warn(
                "The 'datacenter' argument is deprecated and will be removed after 1 July 2026. "
                "Please use the 'location' argument instead. "
                "See https://docs.hetzner.cloud/changelog#2025-12-16-phasing-out-datacenters",
                DeprecationWarning,
                stacklevel=2,
            )
            data["datacenter"] = datacenter.id_or_name
        if location is not None:
            data["location"] = location.id_or_name
        if assignee_id is not None:
            data["assignee_id"] = assignee_id
        if labels is not None:
            data["labels"] = labels

        response = self._client.request(url=self._base_url, json=data, method="POST")

        action = None
        if response.get("action") is not None:
            action = BoundAction(self._parent.actions, response["action"])

        result = CreatePrimaryIPResponse(
            primary_ip=BoundPrimaryIP(self, response["primary_ip"]), action=action
        )
        return result

    def update(
        self,
        primary_ip: PrimaryIP | BoundPrimaryIP,
        auto_delete: bool | None = None,
        labels: dict[str, str] | None = None,
        name: str | None = None,
    ) -> BoundPrimaryIP:
        """Updates the name, auto_delete or labels of a Primary IP.

        :param primary_ip: :class:`BoundPrimaryIP <hcloud.primary_ips.client.BoundPrimaryIP>` or  :class:`PrimaryIP <hcloud.primary_ips.domain.PrimaryIP>`
        :param auto_delete: bool (optional)
               Delete this Primary IP when the resource it is assigned to is deleted
        :param labels: Dict[str, str] (optional)
               User-defined labels (key-value pairs)
        :param name: str (optional)
               New name to set
        :return: :class:`BoundPrimaryIP <hcloud.primary_ips.client.BoundPrimaryIP>`
        """
        data: dict[str, Any] = {}
        if auto_delete is not None:
            data["auto_delete"] = auto_delete
        if labels is not None:
            data["labels"] = labels
        if name is not None:
            data["name"] = name

        response = self._client.request(
            url=f"{self._base_url}/{primary_ip.id}",
            method="PUT",
            json=data,
        )
        return BoundPrimaryIP(self, response["primary_ip"])

    def delete(self, primary_ip: PrimaryIP | BoundPrimaryIP) -> bool:
        """Deletes a Primary IP. If it is currently assigned to an assignee it will automatically get unassigned.

        :param primary_ip: :class:`BoundPrimaryIP <hcloud.primary_ips.client.BoundPrimaryIP>` or  :class:`PrimaryIP <hcloud.primary_ips.domain.PrimaryIP>`
        :return: boolean
        """
        self._client.request(
            url=f"{self._base_url}/{primary_ip.id}",
            method="DELETE",
        )
        # Return always true, because the API does not return an action for it. When an error occurs a HcloudAPIException will be raised
        return True

    def change_protection(
        self,
        primary_ip: PrimaryIP | BoundPrimaryIP,
        delete: bool | None = None,
    ) -> BoundAction:
        """Changes the protection configuration of the Primary IP.

        :param primary_ip: :class:`BoundPrimaryIP <hcloud.primary_ips.client.BoundPrimaryIP>` or  :class:`PrimaryIP <hcloud.primary_ips.domain.PrimaryIP>`
        :param delete: boolean
               If true, prevents the Primary IP from being deleted
        :return: :class:`BoundAction <hcloud.actions.client.BoundAction>`
        """
        data: dict[str, Any] = {}
        if delete is not None:
            data.update({"delete": delete})

        response = self._client.request(
            url=f"{self._base_url}/{primary_ip.id}/actions/change_protection",
            method="POST",
            json=data,
        )
        return BoundAction(self._parent.actions, response["action"])

    def assign(
        self,
        primary_ip: PrimaryIP | BoundPrimaryIP,
        assignee_id: int,
        assignee_type: str = "server",
    ) -> BoundAction:
        """Assigns a Primary IP to a assignee_id.

        :param primary_ip: :class:`BoundPrimaryIP <hcloud.primary_ips.client.BoundPrimaryIP>` or  :class:`PrimaryIP <hcloud.primary_ips.domain.PrimaryIP>`
        :param assignee_id: int
               Assignee the Primary IP shall be assigned to
        :param assignee_type: str
               Assignee the Primary IP shall be assigned to
        :return: :class:`BoundAction <hcloud.actions.client.BoundAction>`
        """
        response = self._client.request(
            url=f"{self._base_url}/{primary_ip.id}/actions/assign",
            method="POST",
            json={"assignee_id": assignee_id, "assignee_type": assignee_type},
        )
        return BoundAction(self._parent.actions, response["action"])

    def unassign(self, primary_ip: PrimaryIP | BoundPrimaryIP) -> BoundAction:
        """Unassigns a Primary IP, resulting in it being unreachable. You may assign it to a server again at a later time.

        :param primary_ip: :class:`BoundPrimaryIP <hcloud.primary_ips.client.BoundPrimaryIP>` or  :class:`PrimaryIP <hcloud.primary_ips.domain.PrimaryIP>`
        :return: :class:`BoundAction <hcloud.actions.client.BoundAction>`
        """
        response = self._client.request(
            url=f"{self._base_url}/{primary_ip.id}/actions/unassign",
            method="POST",
        )
        return BoundAction(self._parent.actions, response["action"])

    def change_dns_ptr(
        self,
        primary_ip: PrimaryIP | BoundPrimaryIP,
        ip: str,
        dns_ptr: str,
    ) -> BoundAction:
        """Changes the dns ptr that will appear when getting the dns ptr belonging to this Primary IP.

        :param primary_ip: :class:`BoundPrimaryIP <hcloud.primary_ips.client.BoundPrimaryIP>` or  :class:`PrimaryIP <hcloud.primary_ips.domain.PrimaryIP>`
        :param ip: str
               The IP address for which to set the reverse DNS entry
        :param dns_ptr: str
               Hostname to set as a reverse DNS PTR entry, will reset to original default value if `None`
        :return: :class:`BoundAction <hcloud.actions.client.BoundAction>`
        """
        response = self._client.request(
            url=f"{self._base_url}/{primary_ip.id}/actions/change_dns_ptr",
            method="POST",
            json={"ip": ip, "dns_ptr": dns_ptr},
        )
        return BoundAction(self._parent.actions, response["action"])
