/*
 *  Device management routines
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../include/driver.h"

int snd_device_new(snd_card_t *card, snd_device_type_t type,
		   void *devstr, int device, snd_device_ops_t *ops,
		   snd_device_t ** rdev)
{
	snd_device_t *dev;

	if (rdev != NULL)
		*rdev = NULL;
	snd_debug_check(card == NULL || devstr == NULL || ops == NULL, -ENXIO);
	dev = (snd_device_t *) snd_kcalloc(sizeof(snd_device_t), GFP_KERNEL);
	if (dev == NULL)
		return -ENOMEM;
	dev->card = card;
	dev->type = type;
	dev->state = SND_DEV_BUILD;
	dev->devstr = devstr;
	dev->number = device;
	dev->ops = ops;
	dev->next = card->devices;
	card->devices = dev;
	if (rdev != NULL)
		*rdev = dev;
	return 0;
}

int snd_device_free(snd_card_t *card, void *devstr)
{
	snd_device_t *dev, *prev;
	
	snd_debug_check(card == NULL || devstr == NULL, -ENXIO);
	prev = NULL;
	dev = card->devices;
	while (dev) {
		if (dev->devstr == devstr) {
			/* unlink */
			if (prev == NULL) {
				card->devices = dev->next;
			} else {
				prev->next = dev->next;
			}
			if (dev->state == SND_DEV_REGISTERED && dev->ops->dev_unregister) {
				if (dev->ops->dev_unregister(devstr))
					snd_printk("device unregister failure\n");
			} else {
				if (dev->ops->dev_free) {
					if (dev->ops->dev_free(devstr))
						snd_printk("device free failure\n");
				}
			}
			if (dev->size > 0 && dev->arg)
				snd_kfree(dev->arg);
			snd_kfree(dev);
			return 0;
		}
		prev = dev;
		dev = dev->next;
	}
	return -ENXIO;
}

int snd_device_register(snd_card_t *card, void *devstr)
{
	snd_device_t *dev, *prev;
	int err;
	
	snd_debug_check(card == NULL || devstr == NULL, -ENXIO);
	prev = NULL;
	dev = card->devices;
	while (dev) {
		if (dev->devstr == devstr) {
			if (dev->state == SND_DEV_BUILD && dev->ops->dev_register) {
				if ((err = dev->ops->dev_register(devstr, dev)) < 0)
					return err;
				dev->state = SND_DEV_REGISTERED;
				return 0;
			}
			return -EBUSY;
		}
		dev = dev->next;
	}
	return -ENXIO;
}

int snd_device_unregister(snd_card_t *card, void *devstr)
{
	return snd_device_free(card, devstr);
}

int snd_device_register_all(snd_card_t *card)
{
	snd_device_t *dev, *prev;
	int err;
	
	snd_debug_check(card == NULL, -ENXIO);
	prev = NULL;
	dev = card->devices;
	while (dev) {
		if (dev->state == SND_DEV_BUILD && dev->ops->dev_register) {
			if ((err = dev->ops->dev_register(dev->devstr, dev)) < 0)
				return err;
			dev->state = SND_DEV_REGISTERED;
		}
		dev = dev->next;
	}
	return 0;
}

int snd_device_free_all(snd_card_t *card)
{
	int err;

	snd_debug_check(card == NULL, -ENXIO);
	while (card->devices) {
		if ((err = snd_device_free(card, card->devices->devstr)) < 0)
			return err;
	}
	return 0;
}
