/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.glib.GObject;
import org.gnu.glib.Type;

/**
 * A ScrolledWindow is a Bin subclass; it's a container that accepts a
 * single child widget.  ScrolledWindow adds scrollbars to the child
 * widget and optionally draws a beveled frame around the child widget.
 * <p>
 * The ScrolledWindow works in two ways.  Some Widgets have native 
 * scrolling support; these widgets have "slots" for {@link Adjustment}
 *  objects.
 * <p>
 * For Widgets that lack native scrolling support the {@link Viewport}
 * Widget acts as an adaptor class, implementing scrollability for 
 * child Widgets that lack their own scrolling capability.
 * <p>
 * If a Widget has native scrolling capabilities it can be added with the
 * <code>add</code> method.  If a Widget does not, you must first add the
 * Widget to a <code>Viewport</code> and then add the <code>Viewport
 * </code> to the ScrolledWindow.  The convenience method <code>
 * addWithViewport</code> does exactly this, so you can ignore the 
 * presence of the <code>Viewport</code>.
 * <p>
 * The position of the scrollbars is controlled by the scroll adjustments.
 * See {@link Adjustment} for details on how to determine the position
 * of the layout.
 */
public class ScrolledWindow extends Bin {

	/**
	 * Create a new ScrolledWindow object.  The two arguments are the
	 * ScrolledWindow's <code>Adjustments</code>; these will be shared
	 * with the scrollbars and the child widgets to keep the bars in
	 * sync with the child. 
	 * 
	 * @param hadj The horizontal Adjustment.
	 * @param vadj The vertical Adjustment.
	 */
	public ScrolledWindow(Adjustment hadj, Adjustment vadj) {
		int horizontal = 0;
		int vertical = 0;
		if (null != hadj) {
			horizontal = hadj.getHandle();
		}
		if (null != vadj) {
			vertical = vadj.getHandle();
		}
		setHandle(ScrolledWindow.gtk_scrolled_window_new(horizontal, vertical));
		initializeEventHandlers();
	}

	/**
	 * Creates a new ScrolledWindow object. This constuctor generates {@link
	 * Adjustment}s automatically.
	 */
	public ScrolledWindow(){
		setHandle(ScrolledWindow.gtk_scrolled_window_new(0, 0));
		initializeEventHandlers();
	}
	
	/**
	 * Construct a ScrolledWindow using a handle to a native resource.
	 */
	public ScrolledWindow(int handle) {
	    super(handle);
		initializeEventHandlers();
	}

	protected void noopInit() {}

	/**
	 * Returns the horizontal Scrollbar's Adjustment.  This can be used
	 * to connect the horizontal Scrollbar to the child Widget's 
	 * horizontal scroll functionality.
	 * 
	 * @return The horizontal Scrollbar's Adjustment.
	 */
	public Adjustment getHAdjustment() {
		int hndl = gtk_scrolled_window_get_hadjustment(getHandle());
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Adjustment)obj;
		return new Adjustment(hndl);
	}

	/**
	 * Sets the horizontal Scrollbar's Adjustment.
	 * 
	 * @param hadj The horizontal Adjustment.
	 */
	public void setHAdjustment(Adjustment hadj) {
		gtk_scrolled_window_set_hadjustment(getHandle(), hadj.getHandle());
	}

	/**
	 * Returns the vertical Scrollbar's Adjustment.  This can be used
	 * to connect the vertical Scrollbar to the child Widget's 
	 * vertical scroll functionality.
	 * 
	 * @return The vertical Scrollbar's Adjustment.
	 */
	public Adjustment getVAdjustment() {
		int hndl = gtk_scrolled_window_get_vadjustment(getHandle());
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Adjustment)obj;
		return new Adjustment(hndl);
	}

	/**
	 * Sets the vertical Scrollbar's Adjustment.
	 * 
	 * @param vadj The vertical Adjustment.
	 */
	public void setVAdjustment(Adjustment vadj) {
		gtk_scrolled_window_set_vadjustment(getHandle(), vadj.getHandle());
	}

	/**
	 * Sets the scrollbar policy for the horizontal and vertical scrollbars.
	 * The policy determines when the scrollbar should appear.
	 * 
	 * @param hScrollBarPolicy The policy for the horizontal ScrollBar.
	 * @param vScrollBarPolicy The policy for the vertical ScrollBar.
	 */
	public void setPolicy(PolicyType hScrollBarPolicy, PolicyType vScrollBarPolicy) {
		ScrolledWindow.gtk_scrolled_window_set_policy(getHandle(), hScrollBarPolicy.getValue(), vScrollBarPolicy.getValue());
	}

	/**
	 * Used to add children without native scrolling capability.  This is
	 * simply a convenience method, it is equivalent to adding the unscrollable
	 * child to a <i>Viewport</i>, then adding the <i>Viewport</i> to the
	 * ScrolledWindow.
	 * 
	 * @param child The Widget to add to the ScrolledWindow.
	 */
	public void addWithViewport(Widget child) {
		ScrolledWindow.gtk_scrolled_window_add_with_viewport(getHandle(), child.getHandle());
	}

	/**
	 * Determines the location of the child widget with respect to the
	 * scrollbars.  The default is <i>CORNER_TOP_LEFT</i>, meaning the
	 * child is in the top left, with the scrollbar underneath and to 
	 * the right.
	 * 
	 * @param windowPlacement The placement for the child widget.
	 */
	public void setPlacement(CornerType windowPlacement) {
		ScrolledWindow.gtk_scrolled_window_set_placement(getHandle(), windowPlacement.getValue());
	}

	/**
	 * Changes the type of shadow drawn around the contents of the
	 * ScrolledWindow.
	 * 
	 * @param type The type of shadow to draw.
	 */
	public void setShadowType(ShadowType type) {
		ScrolledWindow.gtk_scrolled_window_set_shadow_type(getHandle(), type.getValue());
	}

	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gtk_scrolled_window_get_type());
	}

	/****************************************
	 * BEGINNING OF GENERATED CODE
	 ****************************************/
	native static final protected int gtk_scrolled_window_get_type();
	native static final protected int gtk_scrolled_window_new(int hadjustment, int vadjustment);
	native static final protected void gtk_scrolled_window_set_hadjustment(int scrolled_window, int hadjustment);
	native static final protected void gtk_scrolled_window_set_vadjustment(int scrolled_window, int vadjustment);
	native static final protected int gtk_scrolled_window_get_hadjustment(int scrolled_window);
	native static final protected int gtk_scrolled_window_get_vadjustment(int scrolled_window);
	native static final protected void gtk_scrolled_window_set_policy(int scrolled_window, int hscrollbarPolicy, int vscrollbarPolicy);
	native static final protected void gtk_scrolled_window_get_policy(int scrolled_window, int[] hscrollbarPolicy, int[] vscrollbarPolicy);
	native static final protected void gtk_scrolled_window_set_placement(int scrolled_window, int windowPlacement);
	native static final protected int gtk_scrolled_window_get_placement(int scrolled_window);
	native static final protected void gtk_scrolled_window_set_shadow_type(int scrolled_window, int type);
	native static final protected int gtk_scrolled_window_get_shadow_type(int scrolled_window);
	native static final protected void gtk_scrolled_window_add_with_viewport(int scrolled_window, int child);
	/****************************************
	 * END OF GENERATED CODE
	 ****************************************/
}
