/*********************************************************************
 * Copyright (c) 2021 Kichwa Coders Canada Inc. and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
import { Logger, LoggingDebugSession, Event, Handles, Response } from '@vscode/debugadapter';
import { DebugProtocol } from '@vscode/debugprotocol';
import { AmalgamatorClient } from './AmalgamatorClient';
export interface ChildDapArguments {
    /**
     * User visisble name, used to prefix the thread name returned by child dap
     */
    name?: string;
    /**
     * Instead of type/request that VSCode converts to a command using
     * the extension mechanism from the package.json's program settings,
     * the command needs to be fully provided here as runtime/executable.
     *
     * XXX: We read the package.json and use the type/request to create
     * the command, but it is VSCode specific (probably same/similar available
     * in theia?). Look at vscode.debug.onDidStartDebugSession to update the debug
     * config before launch and vscode.extensions.getExtension(...).packageJSON
     * For the former see https://github.com/Microsoft/vscode/issues/32794 and
     * https://github.com/microsoft/vscode-node-debug/blob/a062e12aa1f2307682c0ccb1eb5d99505b1eaea2/src/node/extension/extension.ts#L194
     * for the correct link to the code.
     */
    debugAdapterRuntime: string;
    debugAdapterExecutable: string;
    /**
     * Delay, in milliseconds, before launching this instance. This is being use to demonstrate a small
     * amount of interdependence between launch configs. Defaults to 0.
     */
    delay?: number;
    /**
     * Specify launch or attach request to start debugging for children
     */
    request?: string;
    /**
     * This is the request arguments (normally specified in the launch.json)
     */
    arguments: DebugProtocol.LaunchRequestArguments | DebugProtocol.AttachRequestArguments;
}
export interface RequestArguments extends DebugProtocol.LaunchRequestArguments {
    verbose?: boolean;
    logFile?: string;
    children: ChildDapArguments[];
}
export interface LaunchRequestArguments extends RequestArguments {
}
/**
 * Response for our custom 'cdt-amalgamator/getChildDapNames' request.
 */
export interface ChildDapContents {
    children?: string[];
}
export interface ChildDapResponse extends Response {
    body: ChildDapContents;
}
export declare class StoppedEvent extends Event implements DebugProtocol.StoppedEvent {
    body: {
        reason: string;
        threadId?: number;
        allThreadsStopped?: boolean;
        preserveFocusHint?: boolean;
    };
    constructor(reason: string, threadId: number, preserveFocusHint: boolean, allThreadsStopped: boolean);
}
export declare class ContinuedEvent extends Event implements DebugProtocol.ContinuedEvent {
    body: {
        /** The thread which was continued. */
        threadId: number;
        /** If 'allThreadsContinued' is true, a debug adapter can announce that all threads have continued. */
        allThreadsContinued?: boolean;
    };
    constructor(threadId: number, allThreadsContinued: boolean);
}
export interface ThreadInfo extends DebugProtocol.Thread {
    running: boolean;
}
export declare class AmalgamatorSession extends LoggingDebugSession {
    protected logger: Logger.Logger;
    protected initializeRequestArgs: DebugProtocol.InitializeRequestArguments | undefined;
    protected childDaps: AmalgamatorClient[];
    /**
     * This is a map of the start/end addresses or the instructionPointerReference that client sees -> child DAP index, child DAP addresses
     *
     * It is needed to workaround for problems:
     *  1. VSCode assuming that the instructionPointerReference has the same format as DisassembledInstruction.address
     *  even though the spec doesn't say so.
     *  See the spec at https://microsoft.github.io/debug-adapter-protocol/specification#Types_StackFrame
     *  The problem has been reported at https://github.com/microsoft/vscode/issues/164875
     *  2. VSCode/debug adapter protocol does not support multiple memory spaces.
     *  The problem has been reported at https://github.com/microsoft/vscode/issues/164877
     * Solution:
     *  Based on elements: start addresses or end addresses or the instructionPointerReference to determine
     *  the child dap to be handled.
     * Note:
     *  1. This should be updated after problems are resolved
     *  2. Limit of the solution is this can work incorrectly when child daps have same start addresses
     *  or end addresses or the instructionPointerReference.
     */
    protected addressMap: Map<string, number>;
    protected childDapNames: string[];
    protected childDapIndex?: number;
    protected breakpointHandles: Handles<[AmalgamatorClient, number]>;
    protected frameHandles: Handles<[AmalgamatorClient, number]>;
    protected variableHandles: Handles<[AmalgamatorClient, number]>;
    constructor();
    protected initializeRequest(response: DebugProtocol.InitializeResponse, args: DebugProtocol.InitializeRequestArguments): void;
    protected launchRequest(response: DebugProtocol.LaunchResponse, args: LaunchRequestArguments): Promise<void>;
    protected startAmalgamatorClient(child: ChildDapArguments, index: number): Promise<AmalgamatorClient>;
    protected createChild(child: ChildDapArguments, index: number): Promise<AmalgamatorClient>;
    protected setBreakPointsRequest(response: DebugProtocol.SetBreakpointsResponse, args: DebugProtocol.SetBreakpointsArguments): Promise<void>;
    protected configurationDoneRequest(response: DebugProtocol.ConfigurationDoneResponse, args: DebugProtocol.ConfigurationDoneArguments): Promise<void>;
    /**
     * This is a map of the thread ID that client sees -> child DAP index, child DAP thread id
     */
    protected getThreadInfo(clientThreadId: number): Promise<[number, number]>;
    private threadMapInProcess;
    protected getThreadMap(): Promise<Map<number, [number, number]>>;
    protected getThreadMapInternal(): Promise<[
        Map<number, [number, number]>,
        ThreadInfo[]
    ]>;
    private collectChildTheads;
    protected threadsRequest(response: DebugProtocol.ThreadsResponse): Promise<void>;
    protected stackTraceRequest(response: DebugProtocol.StackTraceResponse, args: DebugProtocol.StackTraceArguments): Promise<void>;
    protected scopesRequest(response: DebugProtocol.ScopesResponse, args: DebugProtocol.ScopesArguments): Promise<void>;
    protected variablesRequest(response: DebugProtocol.VariablesResponse, args: DebugProtocol.VariablesArguments): Promise<void>;
    protected setVariableRequest(response: DebugProtocol.SetVariableResponse, args: DebugProtocol.SetVariableArguments): Promise<void>;
    protected evaluateRequest(response: DebugProtocol.EvaluateResponse, args: DebugProtocol.EvaluateArguments): Promise<void>;
    protected disassembleRequest(response: DebugProtocol.DisassembleResponse, args: DebugProtocol.DisassembleArguments): Promise<void>;
    protected nextRequest(response: DebugProtocol.NextResponse, args: DebugProtocol.NextArguments): Promise<void>;
    protected stepInRequest(response: DebugProtocol.StepInResponse, args: DebugProtocol.StepInArguments): Promise<void>;
    protected stepOutRequest(response: DebugProtocol.StepOutResponse, args: DebugProtocol.StepOutArguments): Promise<void>;
    protected continueRequest(response: DebugProtocol.ContinueResponse, args: DebugProtocol.ContinueArguments): Promise<void>;
    protected pauseRequest(response: DebugProtocol.PauseResponse, args: DebugProtocol.PauseArguments): Promise<void>;
    protected customRequest(command: string, response: DebugProtocol.Response, args: any): Promise<void>;
}
