/**
 * Copyright (c) 2014, 2015 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Christian W. Damus - Initial API and implementation
 */
package org.eclipse.papyrus.uml.profile.assistants.generator;

import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.function.Consumer;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.eclipse.papyrus.infra.gmfdiag.assistant.AssistantFactory;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ConnectionAssistant;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ModelingAssistantProvider;
import org.eclipse.papyrus.infra.gmfdiag.assistant.PopupAssistant;
import org.eclipse.papyrus.infra.types.ElementTypeConfiguration;
import org.eclipse.papyrus.uml.profile.types.generator.Identifiers;
import org.eclipse.papyrus.uml.profile.types.generator.ImpliedExtension;
import org.eclipse.papyrus.uml.profile.types.generator.UML;
import org.eclipse.papyrus.uml.profile.types.generator.UMLElementTypes;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Transformation rule for generating a {@link ModelingAssistantProvider} from a UML {@link Profile}.
 */
@Singleton
@SuppressWarnings("all")
public class ModelingAssistantProviderRule {
  @Extension
  private static AssistantFactory assistantFactory = AssistantFactory.eINSTANCE;

  @Inject
  @Extension
  private Identifiers _identifiers;

  @Inject
  @Extension
  private UML _uML;

  @Inject
  @Extension
  private UMLElementTypes _uMLElementTypes;

  @Inject
  @Extension
  private FiltersUtil _filtersUtil;

  @Inject
  @Extension
  private PopupAssistantRule _popupAssistantRule;

  @Inject
  @Extension
  private ConnectionAssistantRule _connectionAssistantRule;

  public ModelingAssistantProvider toModelingAssistantProvider(final Profile umlProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlProfile);
    final ModelingAssistantProvider _result;
    synchronized (_createCache_toModelingAssistantProvider) {
      if (_createCache_toModelingAssistantProvider.containsKey(_cacheKey)) {
        return _createCache_toModelingAssistantProvider.get(_cacheKey);
      }
      ModelingAssistantProvider _createModelingAssistantProvider = ModelingAssistantProviderRule.assistantFactory.createModelingAssistantProvider();
      _result = _createModelingAssistantProvider;
      _createCache_toModelingAssistantProvider.put(_cacheKey, _result);
    }
    _init_toModelingAssistantProvider(_result, umlProfile);
    return _result;
  }

  private final HashMap<ArrayList<?>, ModelingAssistantProvider> _createCache_toModelingAssistantProvider = CollectionLiterals.newHashMap();

  private void _init_toModelingAssistantProvider(final ModelingAssistantProvider it, final Profile umlProfile) {
    this._identifiers.setIdentifierBase(umlProfile);
    it.setName(umlProfile.getName());
    final Consumer<ImpliedExtension> _function = (ImpliedExtension ext) -> {
      final Function1<ElementTypeConfiguration, String> _function_1 = (ElementTypeConfiguration it_1) -> {
        return this._identifiers.toElementTypeID(ext, it_1);
      };
      Iterable<String> profileElementTypes = IterableExtensions.map(this._uMLElementTypes.getDiagramSpecificElementTypes(ext.getMetaclass()), _function_1);
      final Consumer<String> _function_2 = (String it_1) -> {
        this._filtersUtil.toElementTypeFilter(it_1, umlProfile);
      };
      profileElementTypes.forEach(_function_2);
      boolean _isRelationship = this._uML.isRelationship(this._uMLElementTypes.getEClass(ext.getMetaclass()));
      boolean _not = (!_isRelationship);
      if (_not) {
        Iterables.<String>addAll(it.getElementTypeIDs(), profileElementTypes);
        final Function1<ElementTypeConfiguration, PopupAssistant> _function_3 = (ElementTypeConfiguration it_1) -> {
          return this._popupAssistantRule.toPopupAssistant(ext, it_1);
        };
        Iterables.<PopupAssistant>addAll(it.getPopupAssistants(), IterableExtensions.map(this._uMLElementTypes.getDiagramSpecificElementTypes(ext.getMetaclass()), _function_3));
      } else {
        Iterables.<String>addAll(it.getRelationshipTypeIDs(), profileElementTypes);
        final Function1<ElementTypeConfiguration, ConnectionAssistant> _function_4 = (ElementTypeConfiguration it_1) -> {
          return this._connectionAssistantRule.toConnectionAssistant(ext, it_1);
        };
        Iterables.<ConnectionAssistant>addAll(it.getConnectionAssistants(), IterableExtensions.map(this._uMLElementTypes.getDiagramSpecificElementTypes(ext.getMetaclass()), _function_4));
      }
    };
    this._uML.getAllExtensions(umlProfile).forEach(_function);
  }
}
