/**
 * Copyright (c) 2020 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.ros2.codegen.python.component;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.core.resources.IProject;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.languages.common.base.file.IPFileSystemAccess;
import org.eclipse.papyrus.designer.transformation.base.utils.TransformationException;
import org.eclipse.papyrus.designer.transformation.core.transformations.ExecuteTransformationChain;
import org.eclipse.papyrus.designer.transformation.core.transformations.TransformationContext;
import org.eclipse.papyrus.designer.uml.tools.utils.ElementUtils;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.papyrus.robotics.codegen.common.utils.ActivityUtils;
import org.eclipse.papyrus.robotics.codegen.common.utils.ComponentUtils;
import org.eclipse.papyrus.robotics.core.utils.FunctionUtils;
import org.eclipse.papyrus.robotics.core.utils.InteractionUtils;
import org.eclipse.papyrus.robotics.core.utils.ParameterUtils;
import org.eclipse.papyrus.robotics.core.utils.PortUtils;
import org.eclipse.papyrus.robotics.profile.robotics.commpattern.CommunicationPattern;
import org.eclipse.papyrus.robotics.profile.robotics.components.Activity;
import org.eclipse.papyrus.robotics.profile.robotics.components.ComponentDefinition;
import org.eclipse.papyrus.robotics.profile.robotics.functions.Function;
import org.eclipse.papyrus.robotics.profile.robotics.functions.FunctionKind;
import org.eclipse.papyrus.robotics.ros2.codegen.common.component.AbstractCompTransformations;
import org.eclipse.papyrus.robotics.ros2.codegen.common.component.CallbackGroups;
import org.eclipse.papyrus.robotics.ros2.codegen.common.component.ComponentTransformationUtils;
import org.eclipse.papyrus.robotics.ros2.codegen.common.message.CreateMsgPackage;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.MessageUtils;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.RosHelpers;
import org.eclipse.papyrus.robotics.ros2.codegen.python.utils.ApplyPythonProfile;
import org.eclipse.papyrus.robotics.ros2.codegen.python.utils.ProjectTools;
import org.eclipse.papyrus.robotics.ros2.codegen.python.utils.RosPythonTypes;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.PackageableElement;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLFactory;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;

@SuppressWarnings("all")
public class ComponentTransformations extends AbstractCompTransformations {
  public ComponentTransformations(final IPFileSystemAccess fileAccess, final IProject genProject) {
    super(fileAccess, genProject);
  }

  public boolean createOnConfigureMethod(final org.eclipse.uml2.uml.Class component) {
    boolean _xblockexpression = false;
    {
      Operation op = component.createOwnedOperation("on_configure", null, null);
      Behavior _createOwnedBehavior = component.createOwnedBehavior("on_configure", UMLPackage.eINSTANCE.getOpaqueBehavior());
      OpaqueBehavior ob = ((OpaqueBehavior) _createOwnedBehavior);
      op.getMethods().add(ob);
      op.createOwnedParameter("state", RosPythonTypes.getType(component, "rclpy::lifecycle::State"));
      ob.getLanguages().add("Python");
      EList<String> _bodies = ob.getBodies();
      StringConcatenation _builder = new StringConcatenation();
      {
        List<Activity> _activities = ActivityUtils.getActivities(component);
        for(final Activity activity : _activities) {
          final org.eclipse.uml2.uml.Class activityCl = activity.getBase_Class();
          _builder.newLineIfNotEmpty();
          final String period = ActivityUtils.getPeriod(activity);
          _builder.newLineIfNotEmpty();
          final List<Behavior> activateFcts = FunctionUtils.getFunctions(activity, FunctionKind.ON_CONFIGURE);
          _builder.newLineIfNotEmpty();
          final List<Behavior> periodicFcts = FunctionUtils.getFunctions(activity, FunctionKind.PERIODIC);
          _builder.newLineIfNotEmpty();
          {
            int _size = activateFcts.size();
            boolean _greaterThan = (_size > 0);
            if (_greaterThan) {
              {
                for(final Behavior activateFct : activateFcts) {
                  _builder.append("self.");
                  String _name = activateFct.getName();
                  _builder.append(_name);
                  _builder.append("()");
                  _builder.newLineIfNotEmpty();
                }
              }
            }
          }
          {
            if (((period != null) && (periodicFcts.size() > 0))) {
              _builder.append("# periodic execution (");
              _builder.append(period);
              _builder.append(") for ");
              String _name_1 = activityCl.getName();
              _builder.append(_name_1);
              _builder.append(" using a timer");
              _builder.newLineIfNotEmpty();
              {
                for(final Behavior periodicFct : periodicFcts) {
                  _builder.append("self.timer_");
                  String _name_2 = activityCl.getName();
                  _builder.append(_name_2);
                  _builder.append(" = self.create_timer(");
                  String _convertPeriod = ComponentTransformations.convertPeriod(period);
                  _builder.append(_convertPeriod);
                  _builder.append(", self.");
                  String _name_3 = periodicFct.getName();
                  _builder.append(_name_3);
                  _builder.append(", self.");
                  String _tCallbackGroupName = CallbackGroups.tCallbackGroupName(activityCl);
                  _builder.append(_tCallbackGroupName);
                  _builder.append(")");
                  _builder.newLineIfNotEmpty();
                }
              }
            }
          }
        }
      }
      _builder.newLine();
      _builder.append("return TransitionCallbackReturn.SUCCESS");
      _builder.newLine();
      _xblockexpression = _bodies.add(_builder.toString());
    }
    return _xblockexpression;
  }

  /**
   * Convert a method that uses a unit in seconds or milliseconds to a number
   * passed to the Python call (unlike in C++, there are no ChronoLiterals)
   */
  public static String convertPeriod(final String period) {
    boolean _contains = period.contains("ms");
    if (_contains) {
      float _parseFloat = Float.parseFloat(period.replace("ms", ""));
      final float floatPeriod = (_parseFloat / 1000);
      return Float.valueOf(floatPeriod).toString();
    } else {
      boolean _contains_1 = period.contains("s");
      if (_contains_1) {
        return period.replace("s", "");
      } else {
        return period;
      }
    }
  }

  public boolean createOnActivateMethod(final org.eclipse.uml2.uml.Class component) {
    boolean _xblockexpression = false;
    {
      Operation op = component.createOwnedOperation("on_activate", null, null);
      Behavior _createOwnedBehavior = component.createOwnedBehavior("on_activate", UMLPackage.eINSTANCE.getOpaqueBehavior());
      OpaqueBehavior ob = ((OpaqueBehavior) _createOwnedBehavior);
      ob.getLanguages().add("Python");
      op.getMethods().add(ob);
      op.createOwnedParameter("state", RosPythonTypes.getType(component, "rclpy::lifecycle::State"));
      EList<String> _bodies = ob.getBodies();
      StringConcatenation _builder = new StringConcatenation();
      {
        List<Activity> _activities = ActivityUtils.getActivities(component);
        for(final Activity activity : _activities) {
          final List<Behavior> activateFcts = FunctionUtils.getFunctions(activity, FunctionKind.ON_ACTIVATE);
          _builder.newLineIfNotEmpty();
          {
            int _size = activateFcts.size();
            boolean _greaterThan = (_size > 0);
            if (_greaterThan) {
              {
                for(final Behavior activateFct : activateFcts) {
                  _builder.append("self.");
                  String _name = activateFct.getName();
                  _builder.append(_name);
                  _builder.append("()");
                  _builder.newLineIfNotEmpty();
                }
              }
            }
          }
        }
      }
      _builder.newLine();
      _builder.append("return TransitionCallbackReturn.SUCCESS");
      _builder.newLine();
      _xblockexpression = _bodies.add(_builder.toString());
    }
    return _xblockexpression;
  }

  public boolean createOnDeactivateMethod(final org.eclipse.uml2.uml.Class component) {
    boolean _xblockexpression = false;
    {
      Operation op = component.createOwnedOperation("on_deactivate", null, null);
      Behavior _createOwnedBehavior = component.createOwnedBehavior("on_deactivate", 
        UMLPackage.eINSTANCE.getOpaqueBehavior());
      OpaqueBehavior ob = ((OpaqueBehavior) _createOwnedBehavior);
      ob.getLanguages().add("Python");
      op.getMethods().add(ob);
      op.createOwnedParameter("state", RosPythonTypes.getType(component, "rclpy::lifecycle::State"));
      EList<String> _bodies = ob.getBodies();
      StringConcatenation _builder = new StringConcatenation();
      {
        List<Activity> _activities = ActivityUtils.getActivities(component);
        for(final Activity activity : _activities) {
          final org.eclipse.uml2.uml.Class activityCl = activity.getBase_Class();
          _builder.newLineIfNotEmpty();
          final List<Behavior> deactivateFcts = FunctionUtils.getFunctions(activity, FunctionKind.ON_DEACTIVATE);
          _builder.newLineIfNotEmpty();
          final List<Behavior> periodicFcts = FunctionUtils.getFunctions(activity, FunctionKind.PERIODIC);
          _builder.newLineIfNotEmpty();
          {
            int _size = deactivateFcts.size();
            boolean _greaterThan = (_size > 0);
            if (_greaterThan) {
              {
                for(final Behavior deactivateFct : deactivateFcts) {
                  _builder.append("self.");
                  String _name = deactivateFct.getName();
                  _builder.append(_name);
                  _builder.append("()");
                  _builder.newLineIfNotEmpty();
                }
              }
            }
          }
          {
            int _size_1 = periodicFcts.size();
            boolean _greaterThan_1 = (_size_1 > 0);
            if (_greaterThan_1) {
              {
                for(final Behavior periodicFct : periodicFcts) {
                  _builder.append("self.timer_");
                  String _name_1 = activityCl.getName();
                  _builder.append(_name_1);
                  _builder.append(".cancel()");
                  _builder.newLineIfNotEmpty();
                }
              }
            }
          }
        }
      }
      _builder.newLine();
      _builder.append("return TransitionCallbackReturn.SUCCESS");
      _builder.newLine();
      _xblockexpression = _bodies.add(_builder.toString());
    }
    return _xblockexpression;
  }

  public Parameter createOnCleanupMethod(final org.eclipse.uml2.uml.Class component) {
    Parameter _xblockexpression = null;
    {
      Operation op = component.createOwnedOperation("on_cleanup", null, null);
      Behavior _createOwnedBehavior = component.createOwnedBehavior("on_cleanup", UMLPackage.eINSTANCE.getOpaqueBehavior());
      OpaqueBehavior ob = ((OpaqueBehavior) _createOwnedBehavior);
      ob.getLanguages().add("Python");
      op.getMethods().add(ob);
      _xblockexpression = op.createOwnedParameter("state", RosPythonTypes.getType(component, "rclpy::lifecycle::State"));
    }
    return _xblockexpression;
  }

  public boolean createOnShutdownMethod(final org.eclipse.uml2.uml.Class component) {
    boolean _xblockexpression = false;
    {
      Operation op = component.createOwnedOperation("on_shutdown", null, null);
      Behavior _createOwnedBehavior = component.createOwnedBehavior("on_shutdown", UMLPackage.eINSTANCE.getOpaqueBehavior());
      OpaqueBehavior ob = ((OpaqueBehavior) _createOwnedBehavior);
      ob.getLanguages().add("Python");
      op.getMethods().add(ob);
      op.createOwnedParameter("state", RosPythonTypes.getType(component, "rclpy::lifecycle::State"));
      EList<String> _bodies = ob.getBodies();
      StringConcatenation _builder = new StringConcatenation();
      {
        List<Activity> _activities = ActivityUtils.getActivities(component);
        for(final Activity activity : _activities) {
          final org.eclipse.uml2.uml.Class activityCl = activity.getBase_Class();
          _builder.newLineIfNotEmpty();
          final List<Behavior> shutdownFcts = FunctionUtils.getFunctions(activity, FunctionKind.ON_SHUTDOWN);
          _builder.newLineIfNotEmpty();
          final List<Behavior> periodicFcts = FunctionUtils.getFunctions(activity, FunctionKind.PERIODIC);
          _builder.newLineIfNotEmpty();
          {
            int _size = shutdownFcts.size();
            boolean _greaterThan = (_size > 0);
            if (_greaterThan) {
              {
                for(final Behavior shutdownFct : shutdownFcts) {
                  _builder.append("self.");
                  String _name = shutdownFct.getName();
                  _builder.append(_name);
                  _builder.append("()");
                  _builder.newLineIfNotEmpty();
                }
              }
            }
          }
          {
            int _size_1 = periodicFcts.size();
            boolean _greaterThan_1 = (_size_1 > 0);
            if (_greaterThan_1) {
              {
                for(final Behavior periodicFct : periodicFcts) {
                  _builder.append("self.timer_");
                  String _name_1 = activityCl.getName();
                  _builder.append(_name_1);
                  _builder.append(".destroy()");
                  _builder.newLineIfNotEmpty();
                }
              }
            }
          }
        }
      }
      _builder.newLine();
      _builder.append("return TransitionCallbackReturn.SUCCESS");
      _builder.newLine();
      _xblockexpression = _bodies.add(_builder.toString());
    }
    return _xblockexpression;
  }

  /**
   * Create dependencies for types that are added directly to the code, i.e.
   * don't appear as types
   * Apply Module stereotype
   */
  public void createROS2Dependencies(final org.eclipse.uml2.uml.Class component) {
    try {
      final Type lcTransitionCallbackReturnSC = RosPythonTypes.getType(component, "rclpy::lifecycle::TransitionCallbackReturn");
      component.createDependency(RosPythonTypes.getRclPy(component));
      if ((lcTransitionCallbackReturnSC instanceof Classifier)) {
        component.createDependency(lcTransitionCallbackReturnSC);
      }
      EList<Port> _allPorts = PortUtils.getAllPorts(component);
      for (final Port port : _allPorts) {
        boolean _isQuery = InteractionUtils.isQuery(InteractionUtils.getCommunicationPattern(port));
        if (_isQuery) {
          final Interface msg_type = MessageUtils.getServiceType(port);
          msg_type.setName(ComponentTransformations.getSrvName(RosHelpers.externalName(MessageUtils.getServiceType(port))));
          final org.eclipse.uml2.uml.Package pkg = msg_type.getPackage();
          pkg.setName("srv");
          ApplyPythonProfile.apply(pkg);
          StereotypeUtil.<org.eclipse.papyrus.designer.languages.python.profile.python.Module>applyApp(pkg, org.eclipse.papyrus.designer.languages.python.profile.python.Module.class);
          component.createDependency(msg_type);
        } else {
          boolean _isAction = InteractionUtils.isAction(InteractionUtils.getCommunicationPattern(port));
          if (_isAction) {
          } else {
            final Type msg_type_1 = InteractionUtils.getCommObject(port);
            final org.eclipse.uml2.uml.Package pkg_1 = msg_type_1.getPackage();
            ApplyPythonProfile.apply(pkg_1);
            StereotypeUtil.<org.eclipse.papyrus.designer.languages.python.profile.python.Module>applyApp(pkg_1, org.eclipse.papyrus.designer.languages.python.profile.python.Module.class);
            component.createDependency(msg_type_1);
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public static String getSrvName(final String externalName) {
    final int index = externalName.lastIndexOf("::");
    if ((index != (-1))) {
      return externalName.substring((index + 2));
    }
    return "";
  }

  public static boolean hasProvidedsPort(final org.eclipse.uml2.uml.Class component) {
    EList<Port> _allPorts = PortUtils.getAllPorts(component);
    for (final Port port : _allPorts) {
      {
        final CommunicationPattern cp = InteractionUtils.getCommunicationPattern(port);
        if (((port.getProvideds().size() > 0) && ((InteractionUtils.isPush(cp) || InteractionUtils.isPubSub(cp)) || InteractionUtils.isSend(cp)))) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * Create a parent python package that contains the component's package.
   */
  public boolean createParentPkg(final org.eclipse.uml2.uml.Class component) {
    boolean _xblockexpression = false;
    {
      final org.eclipse.uml2.uml.Package aPkg = UMLFactory.eINSTANCE.createPackage();
      aPkg.setName(this.genProject.getName().toLowerCase());
      _xblockexpression = aPkg.getPackagedElements().add(component.getNearestPackage());
    }
    return _xblockexpression;
  }

  public static void createPubsSubsAttrs(final org.eclipse.uml2.uml.Class component) {
    EList<Port> _allPorts = PortUtils.getAllPorts(component);
    for (final Port port : _allPorts) {
      {
        final CommunicationPattern cp = InteractionUtils.getCommunicationPattern(port);
        if ((InteractionUtils.isPush(cp) || InteractionUtils.isPubSub(cp))) {
          int _size = port.getProvideds().size();
          boolean _greaterThan = (_size > 0);
          if (_greaterThan) {
            final Type rosPublisher = RosPythonTypes.getType(port, "rclpy::lifecycle::Publisher");
            StringConcatenation _builder = new StringConcatenation();
            String _varName = ElementUtils.varName(port);
            _builder.append(_varName);
            _builder.append("_pub_");
            component.createOwnedAttribute(_builder.toString(), rosPublisher);
          } else {
            int _size_1 = port.getRequireds().size();
            boolean _greaterThan_1 = (_size_1 > 0);
            if (_greaterThan_1) {
              final Type rosSubscriber = RosPythonTypes.getType(port, "rclpy::subscription::Subscription");
              StringConcatenation _builder_1 = new StringConcatenation();
              String _varName_1 = ElementUtils.varName(port);
              _builder_1.append(_varName_1);
              _builder_1.append("_sub_");
              component.createOwnedAttribute(_builder_1.toString(), rosSubscriber);
            }
          }
        }
      }
    }
  }

  public static void createSendAttrs(final org.eclipse.uml2.uml.Class component) {
    EList<Port> _allPorts = PortUtils.getAllPorts(component);
    for (final Port port : _allPorts) {
      boolean _isSend = InteractionUtils.isSend(InteractionUtils.getCommunicationPattern(port));
      if (_isSend) {
        int _size = port.getProvideds().size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          final Type rosSubscriber = RosPythonTypes.getType(port, "rclpy::subscription::Subscription");
          StringConcatenation _builder = new StringConcatenation();
          String _varName = ElementUtils.varName(port);
          _builder.append(_varName);
          _builder.append("_recv_");
          component.createOwnedAttribute(_builder.toString(), rosSubscriber);
        } else {
          int _size_1 = port.getRequireds().size();
          boolean _greaterThan_1 = (_size_1 > 0);
          if (_greaterThan_1) {
            final Type rosPublisher = RosPythonTypes.getType(port, "rclpy::lifecycle::Publisher");
            StringConcatenation _builder_1 = new StringConcatenation();
            String _varName_1 = ElementUtils.varName(port);
            _builder_1.append(_varName_1);
            _builder_1.append("_send_");
            component.createOwnedAttribute(_builder_1.toString(), rosPublisher);
          }
        }
      }
    }
  }

  public static void createServiceAttrs(final org.eclipse.uml2.uml.Class component) {
    EList<Port> _allPorts = PortUtils.getAllPorts(component);
    for (final Port port : _allPorts) {
      boolean _isQuery = InteractionUtils.isQuery(InteractionUtils.getCommunicationPattern(port));
      if (_isQuery) {
        int _size = port.getProvideds().size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          final Type rosService = RosPythonTypes.getType(port, "rclpy::service::Service");
          StringConcatenation _builder = new StringConcatenation();
          String _varName = ElementUtils.varName(port);
          _builder.append(_varName);
          _builder.append("_srv_");
          component.createOwnedAttribute(_builder.toString(), rosService);
        } else {
          int _size_1 = port.getRequireds().size();
          boolean _greaterThan_1 = (_size_1 > 0);
          if (_greaterThan_1) {
            final Type rosClient = RosPythonTypes.getType(port, "rclpy::client::Client");
            StringConcatenation _builder_1 = new StringConcatenation();
            String _varName_1 = ElementUtils.varName(port);
            _builder_1.append(_varName_1);
            _builder_1.append("_client_");
            component.createOwnedAttribute(_builder_1.toString(), rosClient);
          }
        }
      }
    }
  }

  /**
   * Add action attributes
   */
  public static void createActionAttrs(final org.eclipse.uml2.uml.Class component) {
    EList<Port> _allPorts = PortUtils.getAllPorts(component);
    for (final Port port : _allPorts) {
      boolean _isAction = InteractionUtils.isAction(InteractionUtils.getCommunicationPattern(port));
      if (_isAction) {
        int _size = port.getProvideds().size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          final Type rosService = RosPythonTypes.getType(port, "rclpy::action::ActionServer");
          component.createDependency(rosService);
          StringConcatenation _builder = new StringConcatenation();
          String _varName = ElementUtils.varName(port);
          _builder.append(_varName);
          _builder.append("_actsrv_");
          component.createOwnedAttribute(_builder.toString(), rosService);
        } else {
          int _size_1 = port.getRequireds().size();
          boolean _greaterThan_1 = (_size_1 > 0);
          if (_greaterThan_1) {
            final Type rosClient = RosPythonTypes.getType(port, "rclpy::action::ActionClient");
            component.createDependency(rosClient);
            StringConcatenation _builder_1 = new StringConcatenation();
            String _varName_1 = ElementUtils.varName(port);
            _builder_1.append(_varName_1);
            _builder_1.append("_actcli_");
            component.createOwnedAttribute(_builder_1.toString(), rosClient);
          }
        }
      }
    }
  }

  /**
   * Remove functions that are not referenced by activities (this can happen after
   * deletion of a function  an activity)
   */
  public static void removeUnrefFunctions(final org.eclipse.uml2.uml.Class component) {
    final ComponentDefinition cd = UMLUtil.<ComponentDefinition>getStereotypeApplication(component, ComponentDefinition.class);
    final ArrayList<Function> fctList = new ArrayList<Function>();
    EList<Activity> _activities = cd.getActivities();
    for (final Activity activity : _activities) {
      fctList.addAll(activity.getFunctions());
    }
    PackageableElement[] _clone = ((PackageableElement[])Conversions.unwrapArray(component.getNearestPackage().getPackagedElements(), PackageableElement.class)).clone();
    for (final PackageableElement pe : _clone) {
      if ((pe instanceof OpaqueBehavior)) {
        final Function fct = UMLUtil.<Function>getStereotypeApplication(pe, Function.class);
        if (((fct != null) && (!fctList.contains(fct)))) {
          ((OpaqueBehavior)pe).destroy();
        }
      }
    }
  }

  public static void createTimerAttrs(final org.eclipse.uml2.uml.Class component) {
    List<Activity> _activities = ActivityUtils.getActivities(component);
    for (final Activity activity : _activities) {
      {
        final String period = ActivityUtils.getPeriod(activity);
        if ((period != null)) {
          NamedElement _qualifiedElementFromRS = ElementUtils.getQualifiedElementFromRS(component, "rclpy::timer::Timer");
          final Type timerBase = ((Type) _qualifiedElementFromRS);
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("timer_");
          String _name = activity.getBase_Class().getName();
          _builder.append(_name);
          _builder.append("_");
          component.createOwnedAttribute(_builder.toString(), timerBase);
        }
      }
    }
  }

  @Override
  public void componentTrafo(final org.eclipse.uml2.uml.Class component, final CreateMsgPackage msgPkgCreator) {
    try {
      MessageUtils.createMessagesOrServices(msgPkgCreator, component);
      this.createParentPkg(component);
      ApplyPythonProfile.apply(component);
      if ((this.genProject == null)) {
        throw new TransformationException(ExecuteTransformationChain.USER_CANCEL);
      }
      this.createOnConfigureMethod(component);
      this.createOnActivateMethod(component);
      this.createOnDeactivateMethod(component);
      this.createOnCleanupMethod(component);
      this.createOnShutdownMethod(component);
      ComponentTransformationUtils.liftFunctions(component);
      Constructor.createConstructor(component);
      this.createROS2Dependencies(component);
      boolean _isRegistered = ComponentUtils.isRegistered(component);
      if (_isRegistered) {
      }
      boolean _hasExternalCode = ActivityUtils.hasExternalCode(component);
      if (_hasExternalCode) {
        new Ros2CodeSkeleton(component).createSkeleton();
      }
      CreateMain.createMain(component);
      ComponentTransformationUtils.removeActivities(component);
      ComponentTransformations.createPubsSubsAttrs(component);
      ComponentTransformations.createSendAttrs(component);
      ComponentTransformations.createServiceAttrs(component);
      ComponentTransformations.createActionAttrs(component);
      ComponentTransformations.createTimerAttrs(component);
      int _size = ParameterUtils.getAllParameters(component).size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        ParameterTransformations.declareParameters(component);
        ParameterTransformations.initParameters(component);
      }
      ParameterTransformations.moveParameters(component);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  @Override
  public void componentCodegen(final org.eclipse.uml2.uml.Class component, final CreateMsgPackage msgPkgCreator) {
    final RoboticsPythonCreator codeGen = new RoboticsPythonCreator(this.genProject, this.fileAccess);
    ComponentTransformationUtils.removeTemplateSig(component);
    ComponentTransformationUtils.removePorts(component);
    ComponentTransformations.removeUnrefFunctions(component);
    TransformationContext.current.project = this.genProject;
    ProjectTools.genCode(codeGen, component);
  }
}
