/**
 * Copyright (c) 2018 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.ros2.codegen.common.utils;

import java.util.Collections;
import java.util.List;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.papyrus.robotics.bpc.profile.bpc.BPCPackage;
import org.eclipse.papyrus.robotics.bpc.profile.bpc.Entity;
import org.eclipse.papyrus.robotics.ros2.preferences.Ros2PreferenceUtils;
import org.eclipse.uml2.uml.util.UMLUtil;

/**
 * Set of utility functions for package creation
 */
@SuppressWarnings("all")
public class PackageXMLUtils {
  /**
   * Return the description of a model from the Entity stereotype
   */
  public static String description(final List<org.eclipse.uml2.uml.Package> modelList) {
    final String description = PackageXMLUtils.getField(modelList, BPCPackage.eINSTANCE.getEntity_Description());
    if ((description != null)) {
      return description;
    }
    return "No description";
  }

  public static String description(final org.eclipse.uml2.uml.Package model) {
    return PackageXMLUtils.description(Collections.<org.eclipse.uml2.uml.Package>singletonList(model));
  }

  /**
   * get author name. If authorship is set and contains a space, assume that it starts with an email address
   * @param modelList a list of models (belonging to the same ROS 2 package) - take
   * 		information from first model with non-null information
   */
  public static String getAuthorName(final List<org.eclipse.uml2.uml.Package> modelList) {
    final String author = PackageXMLUtils.getField(modelList, BPCPackage.eINSTANCE.getEntity_Authorship());
    if ((author != null)) {
      final int idx = author.indexOf(" ");
      if ((idx > 0)) {
        return author.substring((idx + 1));
      }
      return author;
    }
    return Ros2PreferenceUtils.getAuthorName();
  }

  public static String getAuthorName(final org.eclipse.uml2.uml.Package model) {
    return PackageXMLUtils.getAuthorName(Collections.<org.eclipse.uml2.uml.Package>singletonList(model));
  }

  /**
   * get author mail. If authorship is set and contains a space, assume that it starts with an email address
   * @param modelList a list of models (belonging to the same ROS 2 package) - take
   * 		information from first model with non-null information
   */
  public static String getAuthorMail(final List<org.eclipse.uml2.uml.Package> modelList) {
    final String author = PackageXMLUtils.getField(modelList, BPCPackage.eINSTANCE.getEntity_Authorship());
    if ((author != null)) {
      final int idx = author.indexOf(" ");
      if ((idx > 0)) {
        return author.substring(0, idx);
      }
      return author;
    }
    return Ros2PreferenceUtils.getAuthorMail();
  }

  public static String getAuthorMail(final org.eclipse.uml2.uml.Package model) {
    return PackageXMLUtils.getAuthorMail(Collections.<org.eclipse.uml2.uml.Package>singletonList(model));
  }

  /**
   * Get maintainer name. If maintainer is set and contains a space, assume that it starts with an email address
   * Take maintainer information from provenance field in RobMoSys
   * @param modelList a list of models (belonging to the same ROS 2 package) - take
   * 		information from first model with non-null information
   */
  public static String getMaintainerName(final List<org.eclipse.uml2.uml.Package> modelList) {
    final String maintainer = PackageXMLUtils.getField(modelList, BPCPackage.eINSTANCE.getEntity_Provenance());
    if ((maintainer != null)) {
      final int idx = maintainer.indexOf(" ");
      if ((idx > 0)) {
        return maintainer.substring((idx + 1));
      }
      return maintainer;
    }
    return Ros2PreferenceUtils.getMaintainerName();
  }

  public static String getMaintainerName(final org.eclipse.uml2.uml.Package model) {
    return PackageXMLUtils.getMaintainerName(Collections.<org.eclipse.uml2.uml.Package>singletonList(model));
  }

  /**
   * Get maintainer mail. If maintainer is set and contains a space, assume that it starts with an email address
   * Take maintainer information from provenance field in RobMoSys
   * @param modelList a list of models (belonging to the same ROS 2 package) - take
   * 		information from first model with non-null information
   */
  public static String getMaintainerMail(final List<org.eclipse.uml2.uml.Package> modelList) {
    final String maintainer = PackageXMLUtils.getField(modelList, BPCPackage.eINSTANCE.getEntity_Provenance());
    if ((maintainer != null)) {
      final int idx = maintainer.indexOf(" ");
      if ((idx > 0)) {
        return maintainer.substring(0, idx);
      }
      return maintainer;
    }
    return Ros2PreferenceUtils.getMaintainerMail();
  }

  public static String getMaintainerMail(final org.eclipse.uml2.uml.Package model) {
    return PackageXMLUtils.getMaintainerMail(Collections.<org.eclipse.uml2.uml.Package>singletonList(model));
  }

  /**
   * return the first non-null field (feature) from the Entity stereotype
   * from a list of packages
   * 
   * @param modelList
   *            a list of packages (root of models)
   * @param feature
   *            The feature to query
   * @return the author/maintainer/... information
   */
  public static String getField(final List<org.eclipse.uml2.uml.Package> modelList, final EAttribute feature) {
    for (final org.eclipse.uml2.uml.Package pkg : modelList) {
      {
        final Entity application = UMLUtil.<Entity>getStereotypeApplication(pkg, Entity.class);
        if (((application != null) && (application.eGet(feature) instanceof String))) {
          Object _eGet = application.eGet(feature);
          return ((String) _eGet);
        }
      }
    }
    return null;
  }
}
