/**
 * Copyright (c) 2018 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.ros2.codegen.common.message;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.languages.common.base.file.ProjectBasedFileAccess;
import org.eclipse.papyrus.designer.transformation.base.utils.TransformationException;
import org.eclipse.papyrus.designer.transformation.core.transformations.ExecuteTransformationChain;
import org.eclipse.papyrus.designer.transformation.core.transformations.TransformationContext;
import org.eclipse.papyrus.designer.uml.tools.utils.ElementUtils;
import org.eclipse.papyrus.robotics.codegen.common.utils.PackageTools;
import org.eclipse.papyrus.robotics.core.utils.InteractionUtils;
import org.eclipse.papyrus.robotics.profile.robotics.commpattern.CommunicationPattern;
import org.eclipse.papyrus.robotics.ros2.codegen.common.RoboticsTContext;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.MessageUtils;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.RosHelpers;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.ParameterableElement;
import org.eclipse.uml2.uml.TemplateBinding;
import org.eclipse.uml2.uml.TemplateParameterSubstitution;
import org.eclipse.xtext.xbase.lib.Exceptions;

/**
 * Handle creation of a ROS 2 message package
 */
@SuppressWarnings("all")
public class CreateMsgPackage {
  /**
   * List of referenced packages
   */
  private Map<String, Boolean> packageNames;

  /**
   * List of currently available ROS 2 packages
   */
  private Map<String, Boolean> availRosPackages;

  /**
   * Constructor
   */
  public CreateMsgPackage() {
    HashMap<String, Boolean> _hashMap = new HashMap<String, Boolean>();
    this.packageNames = _hashMap;
    this.availRosPackages = MessageUtils.ros2AvailMsgPkgs();
  }

  /**
   * Create message packages for the communication objects referenced by a
   * template binding
   */
  public void createMsgPkgs(final TemplateBinding tb) {
    final CommunicationPattern cp = InteractionUtils.getCommunicationPattern(tb);
    if ((InteractionUtils.isQuery(cp) || InteractionUtils.isAction(cp))) {
      final Interface sd = InteractionUtils.getServiceDefinition(tb);
      this.createMsgPkg(MessageUtils.getMessagePackage(sd));
    }
    EList<TemplateParameterSubstitution> _parameterSubstitutions = tb.getParameterSubstitutions();
    for (final TemplateParameterSubstitution tps : _parameterSubstitutions) {
      {
        ParameterableElement _actual = tps.getActual();
        final DataType actual = ((DataType) _actual);
        MessageUtils.makeDTExternal(actual);
        this.createMsgPkg(MessageUtils.getMessagePackage(actual));
      }
    }
  }

  /**
   * Create a message package for a given data type
   * Convenience function - will obtain the package name
   * from the data type
   */
  public void createMsgPkg(final DataType dt) {
    this.createMsgPkg(MessageUtils.getMessagePackage(dt));
  }

  /**
   * Create a message package for a given package name
   */
  public void createMsgPkg(final org.eclipse.uml2.uml.Package msgPkg) {
    final String msgPkgName = msgPkg.getName();
    boolean _containsKey = this.packageNames.containsKey(msgPkgName);
    if (_containsKey) {
      return;
    }
    this.packageNames.put(msgPkgName, Boolean.valueOf(true));
    NamedElement _qualifiedElementFromRS = ElementUtils.getQualifiedElementFromRS(TransformationContext.current.modelRoot, msgPkgName);
    org.eclipse.uml2.uml.Package srcPkg = ((org.eclipse.uml2.uml.Package) _qualifiedElementFromRS);
    final IProject project = this.getMessageProject(PackageTools.pkgName(msgPkg));
    if ((project == null)) {
      return;
    }
    final ProjectBasedFileAccess fileAccess = new ProjectBasedFileAccess(project);
    RoboticsTContext.current.lst.createMsgBuildFiles(fileAccess, srcPkg);
    final CreateMessage cm = new CreateMessage(this);
    List<DataType> _messages = InteractionUtils.getMessages(srcPkg);
    for (final DataType msg : _messages) {
      cm.generateFile(fileAccess, RosHelpers.escapeUnderscore(msg.getName()), MessageUtils.MESSAGE, cm.createDtOrEnumMsg(msg));
    }
    List<Interface> _queries = InteractionUtils.getQueries(srcPkg);
    for (final Interface sd : _queries) {
      {
        final TemplateBinding tb = InteractionUtils.getTemplateBinding(sd);
        final NamedElement req = MessageUtils.getRequest(tb);
        final NamedElement res = MessageUtils.getResponse(tb);
        cm.generateFile(fileAccess, RosHelpers.escapeUnderscore(InteractionUtils.getNameWoPrefix(sd)), MessageUtils.SERVICE, cm.createServiceMsg(((DataType) req), ((DataType) res)));
      }
    }
    List<Interface> _actions = InteractionUtils.getActions(srcPkg);
    for (final Interface sd_1 : _actions) {
      {
        final TemplateBinding tb = InteractionUtils.getTemplateBinding(sd_1);
        final NamedElement goal = MessageUtils.getGoal(tb);
        final NamedElement res = MessageUtils.getResponse(tb);
        final NamedElement feedback = MessageUtils.getFeedback(tb);
        cm.generateFile(fileAccess, RosHelpers.escapeUnderscore(InteractionUtils.getNameWoPrefix(sd_1)), MessageUtils.ACTION, cm.createActionMsg(((DataType) goal), ((DataType) res), ((DataType) feedback)));
      }
    }
    try {
      IFolder _folder = project.getFolder(MessageUtils.MESSAGE);
      NullProgressMonitor _nullProgressMonitor = new NullProgressMonitor();
      fileAccess.cleanUntouched(_folder, _nullProgressMonitor);
      IFolder _folder_1 = project.getFolder(MessageUtils.SERVICE);
      NullProgressMonitor _nullProgressMonitor_1 = new NullProgressMonitor();
      fileAccess.cleanUntouched(_folder_1, _nullProgressMonitor_1);
      IFolder _folder_2 = project.getFolder(MessageUtils.ACTION);
      NullProgressMonitor _nullProgressMonitor_2 = new NullProgressMonitor();
      fileAccess.cleanUntouched(_folder_2, _nullProgressMonitor_2);
    } catch (final Throwable _t) {
      if (_t instanceof CoreException) {
        final CoreException e = (CoreException)_t;
        throw new RuntimeException(e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }

  /**
   * Obtain the project/package for a given data type (communication object). This
   * function determines the qualified ROS 2 name and checks first, whether the
   * type is already available via "ros2 interface show". If yes, no new project
   * is created
   */
  public IProject getMessageProject(final String msgPkgName) {
    try {
      boolean _containsKey = this.availRosPackages.containsKey(msgPkgName);
      if (_containsKey) {
        return null;
      }
      final IProject genProject = RoboticsTContext.current.lst.getProject(msgPkgName);
      if ((genProject == null)) {
        throw new TransformationException(ExecuteTransformationChain.USER_CANCEL);
      }
      return genProject;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
