/**
 * Copyright (c) 2020 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.robotics.codegen.common.utils;

import java.util.Objects;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.infra.base.StringConstants;
import org.eclipse.papyrus.designer.uml.tools.utils.ConnectorUtil;
import org.eclipse.papyrus.robotics.core.utils.InteractionUtils;
import org.eclipse.uml2.uml.ConnectableElement;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.ConnectorEnd;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.xtend2.lib.StringConcatenation;

/**
 * A set of utility functions around port and topic names
 */
@SuppressWarnings("all")
public class TopicUtils {
  /**
   * Return the topic of a port, currently the port name only
   * 
   * In case of ROS 2, this information will be re-mapped in the launch scripts
   */
  public static String getTopic(final Port port) {
    return port.getName();
  }

  /**
   * Return true, if the name a port has a qualified name, i.e. has /
   * 
   * In case of ROS 2, this information will be re-mapped in the launch scripts
   */
  public static boolean isQualified(final Port port) {
    int _indexOf = port.getName().indexOf(StringConstants.SLASH);
    return (_indexOf != (-1));
  }

  /**
   * Return the topic of a pub/subr port, calculates topic from passed
   * part/port, if provided and from opposite part/port, if required
   */
  public static String getTopic(final Property part, final Port port, final ConnectorEnd oppositeEnd) {
    String _xblockexpression = null;
    {
      ConnectableElement _role = oppositeEnd.getRole();
      final Port oppositePort = ((Port) _role);
      if ((TopicUtils.isQualified(port) && TopicUtils.isQualified(oppositePort))) {
        String _topic = TopicUtils.getTopic(port);
        String _topic_1 = TopicUtils.getTopic(oppositePort);
        boolean _notEquals = (!Objects.equals(_topic, _topic_1));
        if (_notEquals) {
          String _format = String.format(
            "Trying to connect two different qualified port names: %s with %s", 
            TopicUtils.getTopic(port), TopicUtils.getTopic(oppositePort));
          throw new RuntimeException(_format);
        }
      }
      String _xifexpression = null;
      if ((TopicUtils.isQualified(port) || (port.getProvideds().size() > 0))) {
        _xifexpression = TopicUtils.getTopic(part, port);
      } else {
        _xifexpression = TopicUtils.getTopic(oppositeEnd.getPartWithPort(), oppositePort);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  /**
   * Return the topic of a pub/sub port, based on the assumption that the
   * topic is a concatenation of communication object name and port name
   * If the port name is already qualified, use this as topic name without
   * additional qualifications
   */
  public static String getTopic(final Property part, final Port port) {
    String _xifexpression = null;
    boolean _isQualified = TopicUtils.isQualified(port);
    if (_isQualified) {
      _xifexpression = port.getName();
    } else {
      StringConcatenation _builder = new StringConcatenation();
      String _name = part.getName();
      _builder.append(_name);
      _builder.append("/");
      String _name_1 = InteractionUtils.getCommObject(port).getName();
      _builder.append(_name_1);
      _builder.append("/");
      String _name_2 = port.getName();
      _builder.append(_name_2);
      _xifexpression = _builder.toString();
    }
    return _xifexpression;
  }

  /**
   * Return opposite part, based on the assumption that there is a single port
   * targeting a port
   */
  public static ConnectorEnd getOpposite(final org.eclipse.uml2.uml.Class system, final Property part, final Port port) {
    EList<Connector> _ownedConnectors = system.getOwnedConnectors();
    for (final Connector connector : _ownedConnectors) {
      boolean _connectsPort = ConnectorUtil.connectsPort(connector, port);
      if (_connectsPort) {
        final ConnectorEnd end = ConnectorUtil.connEndNotPart(connector, part);
        if ((end != null)) {
          return end;
        }
      }
    }
    return null;
  }
}
