/*****************************************************************************
 * Copyright (c) 2016 Cedric Dumoulin and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Cedric Dumoulin - Initial API and implementation
 *   
 *****************************************************************************/

package org.eclipse.papyrus.designer.languages.java.reverse.ui.dnd;

import java.util.Arrays;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.Request;
import org.eclipse.gmf.runtime.diagram.ui.editparts.GraphicalEditPart;
import org.eclipse.gmf.runtime.diagram.ui.requests.DropObjectsRequest;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.papyrus.designer.languages.java.reverse.ui.ProjectExplorerNodeWalkerWithIProgress;
import org.eclipse.papyrus.designer.languages.java.reverse.ui.ReverseSelectedNodeVisitor;
import org.eclipse.papyrus.designer.languages.java.reverse.ui.ReverseWithJDTParserVisitor;
import org.eclipse.papyrus.designer.languages.java.reverse.ui.dialog.DndReverseCodeDialog;
import org.eclipse.papyrus.designer.languages.java.reverse.ui.dialog.ReverseCodeDialog;
import org.eclipse.papyrus.designer.languages.java.reverse.ui.exception.StopExecutionException;
import org.eclipse.papyrus.designer.languages.java.reverse.ui.utils.QualifiedNamesFromIJavaElementCollector;
import org.eclipse.papyrus.infra.core.resource.NotFoundException;
import org.eclipse.papyrus.infra.core.services.ServiceException;
import org.eclipse.papyrus.infra.core.services.ServicesRegistry;
import org.eclipse.papyrus.infra.core.utils.ServiceUtils;
import org.eclipse.papyrus.infra.gmfdiag.common.helper.NotationHelper;
import org.eclipse.papyrus.infra.gmfdiag.common.utils.ServiceUtilsForEditPart;
import org.eclipse.papyrus.uml.tools.model.UmlModel;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.uml2.uml.NamedElement;

/**
 * @author cedric dumoulin
 *
 */
public class ReverseJobAndTransactionForDrop extends AbstractJobAndTransactionForDrop implements IJobAndTransactionForDrop {

	private static String DefaultGenerationModeleName = "generated"; //$NON-NLS-1$

	private UmlModel umlModel;
	private ServicesRegistry servicesRegistry;
	/**
	 * The selection recorded before the Job was started.
	 */
	private ITreeSelection recordedSelection;


	private ReverseCodeDialog dialog;

	private IProgressMonitor progressMonitor;

	private View parentView;

	private GraphicalEditPart parentViewEditPart;

	private Point firstNodeLocation;

	/**
	 * @see org.eclipse.papyrus.designer.languages.java.reverse.ui.dnd.IJobAndTransactionForDrop#init(org.eclipse.gef.Request, org.eclipse.gef.EditPart)
	 *
	 * @param request
	 * @param targetEditPart
	 */
	@Override
	public void init(Request request, EditPart targetEditPart) throws StopExecutionException {

		// System.err.println(this.getClass().getName() + ".init()");

		super.init(request, targetEditPart);

		// Filter out sources that come from Papyrus Model Explorer.
		List<EObject> sources = getSourceEObjects(request);
		if (!sources.isEmpty()) {
			throw new StopExecutionException();
		}


		try {
			servicesRegistry = ServiceUtilsForEditPart.getInstance().getServiceRegistry(targetEditPart);
			umlModel = (UmlModel) ServiceUtils.getInstance().getModelSet(servicesRegistry).getModel(UmlModel.MODEL_ID);
		} catch (ServiceException e) {
			// Stop command
			throw new StopExecutionException(e);
		}
		// TransactionalEditingDomain editingDomain = getTransactionalEditingDomain(targetEditPart);
		// Compute drop location
		DropObjectsRequest dropRequest = (DropObjectsRequest) request;
		GraphicalEditPart gtEditPart = (GraphicalEditPart) targetEditPart;
		parentViewEditPart = gtEditPart;

		parentView = getTargetView(targetEditPart);

		Point location = dropRequest.getLocation().getCopy();
		gtEditPart.getContentPane().translateToRelative(location);
		gtEditPart.getContentPane().translateFromParent(location);
		location.translate(gtEditPart.getContentPane().getClientArea().getLocation().getNegated());
		firstNodeLocation = location;

		recordedSelection = getCurrentSelection();
	}

	/**
	 * @see org.eclipse.papyrus.designer.languages.java.reverse.ui.dnd.IJobAndTransactionForDrop#canExecute()
	 *
	 * @return true, if it can execute
	 */
	@Override
	public boolean canExecute() {
		return true;
	}

	/**
	 * This method is called after mouse release, and before the Job.
	 *
	 * @throws StopExecutionException
	 */
	@Override
	public void executePreJob() throws StopExecutionException {
		// System.err.println(this.getClass().getName() + ".jobPreprocess()");

		String selectedProjectName = getSelectedProjectName();
		Resource umlResource = umlModel.getResource();
		String modelUID = getResourceUid(umlResource);

		// Open dialog and ask preferences to user.
		// Get reverse parameters from a dialog
		Shell shell = getShell();
		dialog = new DndReverseCodeDialog(shell, modelUID, selectedProjectName, null);

		int res = dialog.open();
		// System.out.println("dialog result =" + res);
		if (res == Window.CANCEL) {
			// Should stop all commands.
			// Return an error code or throw an exception ?
			throw new StopExecutionException("Job canceled by user."); //$NON-NLS-1$
		}

	}

	/**
	 * Compute all variables required for Reverse.
	 *
	 * @param monitor
	 */
	@Override
	public void jobPreprocess(IProgressMonitor monitor) throws StopExecutionException {
		// Prepare parameters
		progressMonitor = monitor;

	}

	/**
	 * @see org.eclipse.papyrus.designer.languages.java.reverse.ui.dnd.IJobAndTransactionForDrop#jobRun(IProgressMonitor)
	 *
	 */
	@SuppressWarnings("unchecked")
	@Override
	public void jobRun(IProgressMonitor monitor) throws StopExecutionException {
		// System.err.println(this.getClass().getName() + ".jobRun()");

		// ProjectExplorerNodeWalker walker = new ProjectExplorerNodeWalker(new DebugProjectExplorerNodeVisitor());
		// walker.visit(getRecordedSelection().toList());

		// Get parameters for reverse
		final List<String> searchPaths = Arrays.asList(dialog.getSearchPath());
		org.eclipse.uml2.uml.Package rootPackage;
		try {
			rootPackage = (org.eclipse.uml2.uml.Package) umlModel.lookupRoot();
		} catch (NotFoundException e) {
			// Should stop all commands.
			// Return an error code or throw an exception ?
			throw new StopExecutionException(e);
		}

		final List<String> creationPaths = dialog.getCreationPaths();

		// Get lists of qualified names of CUs in projects
		QualifiedNamesFromIJavaElementCollector collector = new QualifiedNamesFromIJavaElementCollector(false, true, false);
		List<String> qualifiedNamesInProjects = collector.getQualifiedNamesFromSelection(getRecordedSelection());

		// Perform reverse

		List<NamedElement> returnedReversedNamedElement = null;
		// Do reverse according to selected parser
		switch (dialog.getSelectedParserIndex()) {
		case 0:
		case 1: {
			// JAvaParser
			ReverseSelectedNodeVisitor visitor = new ReverseSelectedNodeVisitor(rootPackage, getPackageName(dialog), searchPaths, creationPaths, qualifiedNamesInProjects);
			ProjectExplorerNodeWalkerWithIProgress reverseWalker = new ProjectExplorerNodeWalkerWithIProgress(visitor);
			reverseWalker.visit(getRecordedSelection().toList(), monitor);

			returnedReversedNamedElement = visitor.getReversedNamedElement();
			break;
		}
		case 2: {
			// JDT Parser
			// System.err.println("Use JDT parser (todo)");

			ReverseWithJDTParserVisitor jdtParserVisitor = new ReverseWithJDTParserVisitor(rootPackage, getPackageName(dialog), searchPaths);
			ProjectExplorerNodeWalkerWithIProgress reverseWalker = new ProjectExplorerNodeWalkerWithIProgress(jdtParserVisitor);
			reverseWalker.visit(getRecordedSelection().toList(), monitor);

			returnedReversedNamedElement = jdtParserVisitor.getReversedNamedElement();
		}
		}

		// Draw reversed NamedElement in diagram

		if (returnedReversedNamedElement != null) {
			final DiagramNodeCreator nodeCreator = new DiagramNodeCreator(parentView, parentViewEditPart, firstNodeLocation);
			nodeCreator.createNodesFor(progressMonitor, returnedReversedNamedElement);
			// Should not run in the main UI, because the transaction will be lost.
		}
	}

	/**
	 * @see org.eclipse.papyrus.designer.languages.java.reverse.ui.dnd.IJobAndTransactionForDrop#jobPostProcess()
	 *
	 */
	@Override
	public void jobPostProcess() throws StopExecutionException {
		// System.err.println(this.getClass().getName() + ".jobPostProcess()");
	}

	/************************************/

	/**
	 * Get the current selection from PlatformUI.getWorkbench() ...
	 * 
	 * @return
	 */
	protected ITreeSelection getCurrentSelection() throws StopExecutionException {
		ISelection selection = null;

		// Try to get selection from ActivePage
		IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
		selection = page.getSelection();
		if (selection instanceof ITreeSelection) {
			return (ITreeSelection) selection;
		}

		throw new StopExecutionException("Can't find a ITreeSelection"); //$NON-NLS-1$
	}

	/**
	 * Get the selection found when the Job was started.
	 * 
	 * @return
	 */
	private ITreeSelection getRecordedSelection() {
		return recordedSelection;
	}


	/**
	 * Get the name of the currently selected project
	 * 
	 * @return the name of the selected project into explorer
	 */
	protected String getSelectedProjectName() {

		// Lookup the java compilation unit.
		TreeSelection treeSelection = (TreeSelection) getRecordedSelection();
		Object firstSelection = treeSelection.getFirstElement();
		if (!(firstSelection instanceof IJavaElement)) {
			return ""; //$NON-NLS-1$
		}

		IJavaElement selectionElement = (IJavaElement) treeSelection.getFirstElement();
		String name = selectionElement.getAncestor(IJavaProject.JAVA_PROJECT).getElementName();
		return name;
	}

	/**
	 * Get an UID identifying uniquely the resource.
	 * This is usually the UTI of the resource.
	 *
	 * @param resource
	 * @return the modelUID name
	 */
	private String getResourceUid(Resource resource) {
		// Try to compute a UID identifying the model. Used to store user settings.
		String modelUID = resource.getURI().toPlatformString(true);
		if (modelUID == null) {
			// Can't compute relative model UID. Use absolute one
			// System.err.println("Can't compute relative model UID. Use absolute one");
			modelUID = resource.getURI().path();
		}
		return modelUID;
	}

	/**
	 * Get the active shell from the active page.
	 * 
	 * @param event
	 * @return
	 */
	protected Shell getShell() {

		return PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().getActivePart().getSite().getShell();
	}

	/**
	 * Find the name of the model provided by the dialog
	 *
	 * @param dialog
	 *            opened dialog to user
	 * @return the name of the model. If the user has changed this name, return the name provided by the user; return the default model name
	 *         otherwise.
	 */
	protected String getPackageName(ReverseCodeDialog dialog) {
		String generationPackageName = dialog.getValue();
		if (generationPackageName == null || generationPackageName.length() == 0) {
			generationPackageName = DefaultGenerationModeleName;
		}
		return generationPackageName;
	}

	/**
	 * Returns the Notation View which is the drop target
	 *
	 * @param targetEditPart
	 *            The drop target edit part
	 * @return
	 * 		The drop target notation View
	 */
	protected View getTargetView(EditPart targetEditPart) {
		return NotationHelper.findView(targetEditPart);
	}
}
