## ----style, echo = FALSE, results = 'asis', warning=FALSE---------------------
suppressPackageStartupMessages(library(bacon))
BiocStyle::markdown()
options(digits=3)

## ----generatedata1------------------------------------------------------------
y <- rnormmix(5000, c(0.9, 0.2, 1.3, 1, 4, 1))

## ----inspectdata1-------------------------------------------------------------
bc <- bacon(y)
bc
estimates(bc)
inflation(bc)
bias(bc)

## ----gsoutput1, fig.cap = "Plot of Gibbs Sampling traces. Each panel represent of one the estimated parameters. Default plot shows the burin-in period as well."----
traces(bc, burnin=FALSE)

## ----gsoutput2, fig.cap = "Gibbs Sampling posterior distributions of two estimated parameters the inflation (sigma 0) and proportion of null features (pi0 0). Posterior plots of the other parameters can be generated by using the `thetas` argument. The ellipical curves corresponding to a 75%, 90% and 95% probability regions for a bivariate normal distribution with mean and covariance estimated form the scatter-plot."----
posteriors(bc)

## ----gsoutput3, fig.cap = "Fit to the data as estimated using the Gibbs Sampling algorithm. Black line represent to overall fit, red the fit of the null distribution and blue and green the alternatives."----
fit(bc, n=100)

## ----plothist1, fig.cap="Histogram of z-scores. With on top standard normal (black) and estimated empirical null distribution (red)."----
plot(bc, type="hist")

## ----plotqq1, fig.cap="Quantile-quantile plot of $-log_{10}$ transformed P-values. Left panel using uncorrected P-values and right panel using bacon bias and inflation corrected P-values."----
plot(bc, type="qq")

## ----simulateddata2, message=FALSE--------------------------------------------
set.seed(12345)
biases <- runif(6, -0.2, 0.2)
inflations <- runif(6, 1, 1.3)
es <- matrix(nrow=5000, ncol=6)
for(i in 1:6)
    es[,i] <- rnormmix(5000, c(0.9, biases[i], inflations[i], 0, 4, 1), shuffle=FALSE)
se <- replicate(6, 0.8*sqrt(4/rchisq(5000,df=4)))
colnames(es) <- colnames(se) <- LETTERS[1:ncol(se)]
rownames(es) <- rownames(se) <- 1:5000
head(rownames(es))
head(colnames(es))

## ----runbacon-----------------------------------------------------------------
library(BiocParallel)
register(MulticoreParam(1, log=TRUE))
bc <- bacon(NULL, es, se)
bc
knitr::kable(estimates(bc))
inflation(bc)
bias(bc)
knitr::kable(tstat(bc)[1:5,])
knitr::kable(pval(bc)[1:5,])
knitr::kable(se(bc)[1:5,])
knitr::kable(es(bc)[1:5,])

## ----gsoutput4, fig.cap = "Plot of Gibbs Sampling traces. Each panel represent of one the estimated parameters. Default plot shows the burin-in period as well."----
traces(bc, burnin=FALSE, index=3)

## ----gsoutput5, fig.cap = "Gibbs Sampling posterior distributions of two estimated parameters the inflation (sigma 0) and proportion of null features (pi0 0). Posterior plots of the other parameters can be generated by using the `thetas` argument. The elliptical curves corresponding to a 75%, 90% and 95% probability regions for a bivariate normal distribution with mean and covariance estimated form the scatter-plot."----
posteriors(bc, index=3)

## ----gsoutput6, fig.cap = "Fit to the data as estimated using the Gibbs Sampling algorithm. Black line represent to overall fit, red the fit of the null distribution and blue and green the alternatives."----
fit(bc, n=100, index=3)

## ----plothist2, fig.cap="Histogram of z-scores. With on top standard normal (black) and estimated empirical null distribution (red)."----
plot(bc, type="hist")

## ----plotqq2, fig.cap="Quantile-quantile plot of $-log_{10}$ transformed P-values. Left panel using uncorrected P-values and right panel using bacon bias and inflation corrected P-values."----
plot(bc, type="qq")

## ----meta---------------------------------------------------------------------
bcm <- meta(bc)
head(pval(bcm))
print(topTable(bcm))

## ----plotqq3, fig.cap="Quantile-quantile plot of $-log_{10}$ transformed P-values for each cohort and the meta-analysis P-values. Left panel using uncorrected P-values and right panel using bacon bias and inflation corrected P-values."----
plot(bcm, type="qq")

## ----simulateddata3, message=FALSE--------------------------------------------
zs <- rnormmix(5000, c(0.9, 0.2, 1.3, 1, 4, 1))

## ----simulateddata4, message=FALSE--------------------------------------------
zs <- cbind(zs, matrix(zs, nrow=5000, ncol=9))
colnames(zs) <- c(paste0("A", 1:10))
rownames(zs) <- 1:5000
head(rownames(zs))
head(colnames(zs))

## ----runbaconseed-------------------------------------------------------------
library(BiocParallel)
register(MulticoreParam(1, log=TRUE))
bc <- bacon(teststatistics = zs, 
            globalSeed = NULL,
            parallelSeed = 42)
bc
head(inflation(bc))
head(bias(bc))

## ----ciave--------------------------------------------------------------------
mean_bias <- mean(bias(bc))
mean_bias
mean_inflation <- mean(inflation(bc))
mean_inflation

## ----cici---------------------------------------------------------------------
CI_bias <- quantile(bias(bc), c(0.025, 0.975))
CI_bias
CI_inflation <- quantile(inflation(bc), c(0.025, 0.975))
CI_inflation

## ----cizspval-----------------------------------------------------------------
avg_corrected_zs <- rowMeans(tstat(bc))
avg_corrected_zs[1:5]
avg_corrected_pvals <- rowMeans(bacon::pval(bc))
avg_corrected_pvals[1:5]

## ----session info, echo=FALSE-------------------------------------------------
sessionInfo()

