## ----setup, include = FALSE---------------------------------------------------
knitr::opts_knit$set(
    root.dir = dirname(getwd())
)

knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>"
)

## ----init, message = FALSE, echo = FALSE, results = "hide"--------------------
library(BiocStyle)
library(lcmsPlot)

## ----install, eval=FALSE------------------------------------------------------
# if (!requireNamespace("BiocManager", quietly = TRUE))
#     install.packages("BiocManager")
# BiocManager::install("lcmsPlot")

## ----load-data, message=FALSE-------------------------------------------------
library(lcmsPlot)

exdir <- tempdir()

# Unzip the file containing the mzML files
utils::unzip(
    system.file("extdata", "standards-mzml.zip", package = "lcmsPlot"),
    exdir = exdir
)

mzml_dir <- file.path(exdir, "mzml")

samples_metadata <- data.frame(
    sample_id = c(
        "l-proline-MS1",
        "l-proline-MS2",
        "l-kynurenine-MS1",
        "l-kynurenine-MS2"),
    sample_name = c(rep("L-Proline", 2), rep("L-Kynurenine", 2)),
    ms_level = c(1, 2, 1, 2)
)

## ----data-overview------------------------------------------------------------
ms1_samples_metadata <- samples_metadata |> dplyr::filter(ms_level == 1)
ms1_raw_files <- file.path(
    mzml_dir,
    paste0(ms1_samples_metadata$sample_id, ".mzML"))

lcmsPlot(ms1_raw_files, metadata = ms1_samples_metadata) +
    lp_chromatogram(
        features = data.frame(
            sample_id = c("l-proline-MS1", "l-kynurenine-MS1"),
            mz = c(116.0703793, 209.091824),
            rt = c(425.74, 365.72)
        ),
        ppm = 5,
        rt_tol = 5
    ) +
    lp_facets(facets = "sample_name", free_x = TRUE, ncol = 2)

## ----plot-chromatograms-with-spectra------------------------------------------
ms2_samples_metadata <- samples_metadata |> dplyr::filter(ms_level == 2)
ms2_raw_files <- file.path(
    mzml_dir,
    paste0(ms2_samples_metadata$sample_id, ".mzML"))

lcmsPlot(ms2_raw_files, metadata = ms2_samples_metadata) +
    lp_chromatogram(
        features = data.frame(
            sample_id = c("l-proline-MS2", "l-kynurenine-MS2"),
            mz = c(116.0703793, 209.091824),
            rt = c(425.74, 365.72)
        ),
        ppm = 5,
        rt_tol = 5
    ) +
    lp_spectra(
        rt = c("l-proline-MS2" = 425.74, "l-kynurenine-MS2" = 365.72),
        mode = "closest",
        ms_level = 2) +
    lp_facets(facets = "sample_name", free_x = TRUE, ncol = 2)

## ----custom-theme-------------------------------------------------------------
my_custom_theme <- function() {
    ggplot2::theme(
        # Title
        plot.title = ggplot2::element_text(
            size = 16,
            margin = ggplot2::margin(b = 15)),
        
        # Axis labels
        axis.title = ggplot2::element_text(
            size = 11,
            face = "bold",
            color = "#1d293d"),
        
        # Facet strip
        strip.text = ggplot2::element_text(
            color = "#1d293d",
            size = 9,
            hjust = 0,
            margin = ggplot2::margin(t = 3, r = 10, b = 6, l = 10)
        ),
        strip.background = ggplot2::element_rect(
            color = "#90a1b9",
            fill = "#e2e8f0",
            linewidth = 0.8),
        
        # Box around the panel
        panel.border = ggplot2::element_rect(
            color = "#90a1b9",
            fill = NA,
            linewidth = 1),
        
        # Customize grid lines inside the plot
        panel.grid.major = ggplot2::element_line(
            color = "#e2e8f0",
            linewidth = 0.5),
        panel.grid.minor = ggplot2::element_line(
            color = "#e2e8f0",
            linewidth = 0.5)
    )
}

## ----plot-with-custom-theme---------------------------------------------------
lcmsPlot(ms1_raw_files, metadata = ms1_samples_metadata) +
    lp_chromatogram(
        features = data.frame(
            sample_id = c("l-proline-MS1", "l-kynurenine-MS1"),
            mz = c(116.0703793, 209.091824),
            rt = c(425.74, 365.72)
        ),
        ppm = 5,
        rt_tol = 5
    ) +
    lp_facets(facets = "sample_name", free_x = TRUE, ncol = 2) +
    lp_get_plot() +
    my_custom_theme()

## ----cleanup, echo=FALSE, results='hide', message=FALSE, warning=FALSE--------
unlink(mzml_dir, recursive = TRUE, force = TRUE)

## ----session-info-------------------------------------------------------------
sessionInfo()

