﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/s3/S3_EXPORTS.h>
#include <aws/s3/model/Event.h>
#include <aws/s3/model/NotificationConfigurationFilter.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace S3 {
namespace Model {

/**
 * <p>A container for specifying the configuration for Lambda
 * notifications.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/s3-2006-03-01/LambdaFunctionConfiguration">AWS
 * API Reference</a></p>
 */
class LambdaFunctionConfiguration {
 public:
  AWS_S3_API LambdaFunctionConfiguration() = default;
  AWS_S3_API LambdaFunctionConfiguration(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_S3_API LambdaFunctionConfiguration& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_S3_API void AddToNode(Aws::Utils::Xml::XmlNode& parentNode) const;

  ///@{

  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  LambdaFunctionConfiguration& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Lambda function that Amazon S3 invokes
   * when the specified event type occurs.</p>
   */
  inline const Aws::String& GetLambdaFunctionArn() const { return m_lambdaFunctionArn; }
  inline bool LambdaFunctionArnHasBeenSet() const { return m_lambdaFunctionArnHasBeenSet; }
  template <typename LambdaFunctionArnT = Aws::String>
  void SetLambdaFunctionArn(LambdaFunctionArnT&& value) {
    m_lambdaFunctionArnHasBeenSet = true;
    m_lambdaFunctionArn = std::forward<LambdaFunctionArnT>(value);
  }
  template <typename LambdaFunctionArnT = Aws::String>
  LambdaFunctionConfiguration& WithLambdaFunctionArn(LambdaFunctionArnT&& value) {
    SetLambdaFunctionArn(std::forward<LambdaFunctionArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon S3 bucket event for which to invoke the Lambda function. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html">Supported
   * Event Types</a> in the <i>Amazon S3 User Guide</i>.</p>
   */
  inline const Aws::Vector<Event>& GetEvents() const { return m_events; }
  inline bool EventsHasBeenSet() const { return m_eventsHasBeenSet; }
  template <typename EventsT = Aws::Vector<Event>>
  void SetEvents(EventsT&& value) {
    m_eventsHasBeenSet = true;
    m_events = std::forward<EventsT>(value);
  }
  template <typename EventsT = Aws::Vector<Event>>
  LambdaFunctionConfiguration& WithEvents(EventsT&& value) {
    SetEvents(std::forward<EventsT>(value));
    return *this;
  }
  inline LambdaFunctionConfiguration& AddEvents(Event value) {
    m_eventsHasBeenSet = true;
    m_events.push_back(value);
    return *this;
  }
  ///@}

  ///@{

  inline const NotificationConfigurationFilter& GetFilter() const { return m_filter; }
  inline bool FilterHasBeenSet() const { return m_filterHasBeenSet; }
  template <typename FilterT = NotificationConfigurationFilter>
  void SetFilter(FilterT&& value) {
    m_filterHasBeenSet = true;
    m_filter = std::forward<FilterT>(value);
  }
  template <typename FilterT = NotificationConfigurationFilter>
  LambdaFunctionConfiguration& WithFilter(FilterT&& value) {
    SetFilter(std::forward<FilterT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;

  Aws::String m_lambdaFunctionArn;

  Aws::Vector<Event> m_events;

  NotificationConfigurationFilter m_filter;
  bool m_idHasBeenSet = false;
  bool m_lambdaFunctionArnHasBeenSet = false;
  bool m_eventsHasBeenSet = false;
  bool m_filterHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3
}  // namespace Aws
